/* **********************************************************
 * Copyright 2013, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings.client;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;

/**
 * Client-side policy which decides what to do when an error happens during an
 * API invocation.
 * <p>
 * This policy can be used it to automatically recover from temporary problems
 * like expired sessions, service unavailable, network issues.
 */
public interface RetryPolicy {

    /**
     * Context about an invocation. Describes the invocation attempt that caused
     * the error.
     */
    public final class RetryContext {

        private final String serviceId;
        private final String operationId;
        private final ExecutionContext executionContext;
        private final StructValue input;

        /**
         * Creates a context.
         *
         * @param serviceId service fully-qualified name; must not be
         *                  <code>null</code> or empty
         * @param operationId operation name; must not be <code>null</code> or
         *                    empty
         * @param executionContext execution context; must not be
         *                         <code>null</code>
         * @param input input value; must not be <code>null</code>
         */
        public RetryContext(String serviceId,
                String operationId,
                ExecutionContext executionContext,
                StructValue input) {
            Validate.notEmpty(serviceId);
            Validate.notEmpty(operationId);
            Validate.notNull(executionContext);
            Validate.notNull(input);
            this.serviceId = serviceId;
            this.operationId = operationId;
            this.executionContext = executionContext;
            this.input = input;
        }

        /**
         * @return fully-qualified service name of the invocation
         */
        public String getServiceId() {
            return serviceId;
        }

        /**
         * @return operation name of the invocation
         */
        public String getOperationId() {
            return operationId;
        }

        /**
         * @return execution context of the invocation
         */
        public ExecutionContext getExecutionContext() {
            return executionContext;
        }

        /**
         * @return input value of the invocation
         */
        public StructValue getInput() {
            return input;
        }
    }

    /**
     * Retry specification. Describes how to retry an invocation.
     */
    public final class RetrySpec {

        private final InvocationConfig invocationConfig;

        /**
         * Creates a specification, which may have an {@link InvocationConfig}.
         * The {@link InvocationConfig} will be used when retrying the
         * invocation.
         *
         * @param invocationConfig configuration for the next invocation
         *                         attempt; if <code>null</code> the original
         *                         invocation configuration will be used
         */
        public RetrySpec(InvocationConfig invocationConfig) {
            this.invocationConfig = invocationConfig;
        }

        /**
         * @return configuration for the next invocation attempt; can be
         *         <code>null</code>
         */
        public InvocationConfig getInvocationConfig() {
            return invocationConfig;
        }
    }

    /**
     * An invocation error has occurred. The policy should decide whether to
     * retry the invocation or let it fail.
     * <p>
     * The error can be any exception that can be reported by the stub. See
     * {@link AsyncCallback#onError(RuntimeException)} for details on possible
     * errors.
     * <p>
     * The method is expected to return a retry specification. This
     * specification will be used to retry the invocation. If the method
     * returns <code>null</code> or throws a {@link RuntimeException}, there
     * will be no further attempts and the invocation will fail.
     *
     * @param error invocation error; must not be </code>null</code>
     * @param retryContext context about the invocation error; must not be <code>null</code>
     * @param invocationAttempt number of the invocation attempt; first attempt
     *                          is 0, second attempt (after error on retry) is 1, etc.
     * @return retry specification; <code>null</code> means no retry
     */
    RetrySpec onInvocationError(RuntimeException error,
                                RetryContext retryContext,
                                int invocationAttempt);
}
