/* **********************************************************
 * Copyright 2012 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings.server;

import com.vmware.vapi.std.Progress;

/**
 * <p>
 * Context of an asynchronous operation. Combines useful information about the
 * operation (see {@link #getInvocationContext()}) with means to complete the
 * operation (with a result or an error) or report progress.
 * </p>
 *
 * <p>
 * This context is the only legal way to complete an operation invocation (with
 * a result object or an error) or report progress. The invocation will hang
 * until either {@link #setResult(Object)} or
 * {@link #setError(RuntimeException)} marks the operation as complete.
 * Completing the operation more than once is forbidden. Reporting progress
 * after the operation is complete is forbidden.
 * </p>
 */
public abstract class AsyncContext<T> {

    /**
     * Returns context information about the method invocation.
     *
     * @return context of the method invocation
     */
    public abstract InvocationContext getInvocationContext();

    /**
     * Reports progress for the operation invocation. May be invoked multiple
     * times. Must not be invoked after {@link #setResult(Object)} or
     * {@link #setError(RuntimeException)} is called.
     *
     * @param progress  progress status of the operation invocation; must not
     *                  be <code>null</code>
     */
    public abstract void updateProgress(Progress progress);

    /**
     * Completes the invocation with a result object. Must not be called after
     * the operation is already completed (with either a result object or an
     * error).
     *
     * @param result  operation result
     */
    public abstract void setResult(T result);

    /**
     * Completes the invocation with an error. Must not be called after the
     * operation is already completed (with either a result object or an error).
     *
     * @param error  operation error
     */
    public abstract void setError(RuntimeException error);
}
