/* **********************************************************
 * Copyright 2012-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.core;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.data.DataValue;

/**
 * This utility provides a synchronous view of an {@link AsyncHandle}.
 */
public class AsyncHandleSyncAdapter<T> extends AsyncHandle<T> {
    private static final Logger LOGGER = LoggerFactory
            .getLogger(AsyncHandleSyncAdapter.class);

    private final Lock lock = new ReentrantLock();
    private final Condition set = lock.newCondition();
    private boolean completed = false;

    private T result;
    private RuntimeException error;

    /**
     * Waits until this handle is set either an output or an error value.
     */
    private void await() {
        lock.lock();
        try {
            while (!completed) {
                set.await();
            }
        } catch (InterruptedException ex) {
            Thread.currentThread().interrupt();
        } finally {
            lock.unlock();
        }
    }

    /**
     * Returns the result or throws the error of this async handle.
     *
     * @return the result, if set by {@link #setResult(Object)}
     * @throws RuntimeException the error, if set by
     *         {@link #setError(RuntimeException)}
     */
    public T get() {
        await();
        if (error != null) {
            throw error;
        }
        return result;
    }

    @Override
    public void setResult(T result) {
        tryReadHttpTerminalFrame(result);
        lock.lock();
        try {
            assertNotCompleted();
            this.result = result;
            this.completed = true;
            set.signalAll();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void setError(RuntimeException value) {
        lock.lock();
        try {
            assertNotCompleted();
            this.error = value;
            this.completed = true;
            set.signalAll();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void updateProgress(DataValue progress) {
        lock.lock();
        try {
            assertNotCompleted();
        } finally {
            lock.unlock();
        }
    }

    /**
     * Verifies that the operation is not yet completed.
     *
     * @throws IllegalArgumentException if the operation is already completed
     */
    private void assertNotCompleted() {
        if (completed) {
            throw new IllegalStateException("Operation is already completed");
        }
    }

    /*
     * This method makes sure that HTTP streaming responses are fully consumed
     * and released properly when using blocking stub methods. Since the
     * transport is not aware of the binding method semantics it will block
     * reading and releasing stream responses until demand is expressed by the
     * client. In case of mono response method we make sure to unblock the
     * transport so it can release the HTTP response. Therefore, we ought to
     * initiate a read, which will go through the HTTP terminal frame, hence,
     * closing the underlying HTTP response.
     */
    private void tryReadHttpTerminalFrame(T methodResult) {
        try {
            if (methodResult instanceof MethodResult) {
                LOGGER.debug("Attempting to close the TCP connection.");
                MethodResult result = (MethodResult) methodResult;
                if (result.getNext() != null) {
                    LOGGER.debug("Reading HTTP terminal frame.");
                    // <null> is supplied to forbid any further readings.
                    result.getNext().accept(null);
                }
            }
        } catch (Exception e) {
            LOGGER.warn("Failed to read HTTP terminal frame. Connection not closed.",
                        e);
        }
    }
}
