/* **********************************************************
 * Copyright 2011-2013, 2019, 2021 VMware, Inc. All rights reserved.
 *      -- VMware Confidential
 * *********************************************************
 */

package com.vmware.vapi.core;

import com.vmware.vapi.internal.util.Validate;


/**
 * The <code>MethodIdentifier</code> class identifies an
 * <code>{@link com.vmware.vapi.provider.ApiMethod}</code>
 * instance.
 */
public class MethodIdentifier {

    private static final String METHOD_NAME_DELIMITER = ".";

    private final InterfaceIdentifier iface;
    private final String method;

    /**
     * Creates a method identifier given a fully-qualified name
     * ([interface].[method]).
     *
     * @param path  fully-qualified method name. must not be <code>null</code>.
     * @throws IllegalArgumentException if <code>path</code> is <code>null</code>
     */
    public MethodIdentifier(String path) {
        Validate.notNull(path);
        int index = path.lastIndexOf(METHOD_NAME_DELIMITER);
        if (index == 0) {
            this.iface = new InterfaceIdentifier("");
            this.method = "";
        } else if (index > 0) {
            this.iface = new InterfaceIdentifier(path.substring(0, index));
            this.method = path.substring(index + 1);
        } else {
            this.iface = new InterfaceIdentifier("");
            this.method = path;
        }
    }

    /**
     * Creates a method identifier given an interface identifier
     * and a method name.
     *
     * @param iface   interface identifier
     * @param method  method name
     * @throws IllegalArgumentException if <code>iface</code> or
     *         <code>method</code> is <code>null</code>
     */
    public MethodIdentifier(InterfaceIdentifier iface, String method) {
        Validate.notNull(iface);
        Validate.notNull(method);
        this.iface = iface;
        this.method = method;
    }

    /**
     * Returns the name of this method.
     *
     * @return  method name
     */
    public String getName() {
        return method;
    }

    /**
     * Returns the identifier of the interface containing this method.
     *
     * @return  interface identifier of the interface containing this method
     */
    public InterfaceIdentifier getInterfaceIdentifier() {
        return iface;
    }

    /**
     * @return the fully-qualified name ([interface].[method]).
     */
    public String getFullyQualifiedName() {
        return getFullyQualifiedName(iface.getName(), method);
    }

    public static String getFullyQualifiedName(String iface, String method) {
        return iface + METHOD_NAME_DELIMITER + method;
    }

    /** {@inheritDoc} */
    @Override
    public boolean equals(Object o) {
        if (o instanceof MethodIdentifier) {
            MethodIdentifier tmp = (MethodIdentifier) o;
            return method.equals(tmp.method) && iface.equals(tmp.iface);
        }
        return false;
    }

    /** {@inheritDoc} */
    @Override
    public int hashCode() {
        return iface.hashCode() + method.hashCode();
    }

    /** {@inheritDoc} */
    @Override
    public String toString() {
        return iface + "." + method;
    }
}