/* **********************************************************
 * Copyright 2013 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.data;

import java.util.LinkedList;
import java.util.List;

import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;


/**
 * <code>DataDefinition</code> for <code>Exception</code> IDL type.
 */
public class AnyErrorDefinition extends DataDefinition {
    private static final AnyErrorDefinition INSTANCE =
            new AnyErrorDefinition();

    public static AnyErrorDefinition getInstance() {
        return INSTANCE;
    }

    /**
     * Constructor.
     *
     * <p>Private one, since this class is a Singleton.
     */
    private AnyErrorDefinition() {
    }

    @Override
    public DataType getType() {
        return DataType.ANY_ERROR;
    }

    /**
     * Validates that the specified {@link DataValue} is an instance of this
     * data definition.
     *
     * <p>Validates that supplied <code>value</code> is not <code>null</code>
     * and it's type matches {@link DataType#ERROR}.
     *
     * @param value  the <code>DataValue</code> to validate
     * @return       a list of messages describing validation problems; empty
     *               list indicates that validation was successful
     */
    @Override
    public List<Message> validate(DataValue value) {
        List<Message> result = new LinkedList<Message>();
        if (value == null) {
            result.add(MessageFactory.getMessage("vapi.data.validate.mismatch",
                                                 getType().toString(),
                                                 "null"));
        } else if (value.getType() != DataType.ERROR) {
            result.add(MessageFactory.getMessage("vapi.data.validate.mismatch",
                                                 getType().toString(),
                                                 value.getType().toString()));
        }
        return result;
    }

    @Override
    public void accept(DefinitionVisitor visitor) {
        visitor.visit(this);
    }
}
