/* **********************************************************
 * Copyright 2011-2014 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * ListDefinition.java --
 *
 *      Type definition for vAPI list built-in type
 */

package com.vmware.vapi.data;

import java.util.*;
import com.vmware.vapi.*;

public class ListDefinition extends DataDefinition {
    /**
     * The Definition of the type that is contained in a list.
     */
    private DataDefinition elementType;

    public ListDefinition(DataDefinition elementType) {
        if (elementType == null) {
            throw new IllegalArgumentException("List definition requires element definition.");
        }
        this.elementType = elementType;
    }

    public DataType getType() {
        return DataType.LIST;
    }

    public DataDefinition getElementType() {
        return this.elementType;
    }

    public ListValue newInstance() {
        return new ListValue();
    }

    /**
     * {@inheritDoc}
     *
     * <p>In addition, validates that all entries from the list value
     * expected type for <code>this</code> definition
     */
    @Override
    public List<Message> validate(DataValue value) {
        List<Message> errors = super.validate(value);

        if (!errors.isEmpty()) {
            return errors;
        }

        // this is safe because super.validate() verified the type
        ListValue list = (ListValue) value;

        int index = 0;
        for (DataValue elem : list) {
            List<Message> subErrors = elementType.validate(elem);

            if (!subErrors.isEmpty()) {
                errors.add(MessageFactory.getMessage(
                        "vapi.data.list.invalid.entry",
                        elem.toString(),
                        String.valueOf(index)));

                errors.addAll(subErrors);

                // stop processing at first detected error
                break;
            }
            index++;
        }

        return errors;
    }

    @Override
    public void completeValue(DataValue value) {
        // completing list value
        if (value != null && value.getType() == DataType.LIST) {
            ListValue list = (ListValue) value;

            for (DataValue elem : list) {
                elementType.completeValue(elem);
            }
        }
    }

    @Override
    public boolean equals(Object other) {
        if (this == other) {
            return true;
        }

        if (!(other instanceof ListDefinition)) {
            return false;
        }
        ListDefinition otherList = (ListDefinition) other;
        return this.elementType.equals(otherList.elementType);
    }

    @Override
    public int hashCode() {
        return 17 + 31 * elementType.hashCode();
    }

    @Override
    public String toString() {
        return "List[" + elementType.toString() + "]";
    }

    @Override
    public void accept(DefinitionVisitor visitor) {
        visitor.visit(this);
    }
}
