/* **********************************************************
 * Copyright 2012, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.data;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.util.List;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.Message;
import com.vmware.vapi.internal.util.Validate;

/**
 * Reference to a {@link StructDefinition}. If the reference is resolved, it is
 * bound to a specific {@link StructDefinition} target. If the reference is
 * unresolved, its target is <code>null</code>.
 *
 * <p>
 * <i>Thread-safety:</i> Conditionally thread-safe. Resolving the reference via
 * {@link #setTarget(StructDefinition)} is not thread-safe.
 * </p>
 */
public class StructRefDefinition extends DataDefinition {

    private final String name;
    private StructDefinition structDefinition;

    /**
     * Creates an unresolved reference.
     *
     * @param name structure name; must not be <code>null</code>
     */
    public StructRefDefinition(String name) {
        Validate.notNull(name);
        this.name = name;
    }

    /**
     * Creates a resolved reference.
     *
     * @param structDefinition structure definition; must not be
     *                         <code>null</code>
     */
    public StructRefDefinition(StructDefinition structDefinition) {
        Validate.notNull(structDefinition);
        this.name = structDefinition.getName();
        this.structDefinition = structDefinition;
    }

    /**
     * Returns the structure name of this reference.
     *
     * @return structure name; never <code>null</code>
     */
    public String getName() {
        return name;
    }

    /**
     * Returns the target structure definition of this reference.
     *
     * @return the target of this reference; will be <code>null</code> for an
     *         unresolved reference
     */
    public StructDefinition getTarget() {
        return structDefinition;
    }

    /**
     * Resolves the reference. An unresolved reference can be resolved exactly
     * once. A resolved reference cannot be re-resolved.
     *
     * @param structDefinition structure definition; must not be
     *                         <code>null</code>
     * @throws CoreException if the reference is already resolved (already has
     *                       a target) or if the name of the reference does not
     *                       match the name of the definition
     */
    public void setTarget(StructDefinition structDefinition) {
        Validate.notNull(structDefinition);
        if (this.structDefinition != null) {
            throw new CoreException(getMessage(
                    "vapi.data.structref.already.resolved", name));
        }
        if (!name.equals(structDefinition.getName())) {
            throw new CoreException(getMessage(
                    "vapi.data.structref.resolve.type.mismatch",
                    name,
                    structDefinition.getName()));
        }
        this.structDefinition = structDefinition;
    }


    @Override
    public DataType getType() {
        return DataType.STRUCTURE_REF;
    }

    @Override
    public void accept(DefinitionVisitor visitor) {
        visitor.visit(this);
    }

    @Override
    public List<Message> validate(DataValue value) {
        checkResolved();
        return structDefinition.validate(value);
    }

    @Override
    public void completeValue(DataValue value) {
        checkResolved();
        structDefinition.completeValue(value);
    }

    private void checkResolved() {
        if (structDefinition == null) {
            throw new CoreException(getMessage(
                    "vapi.data.structref.not.resolved", name));
        }
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!(obj instanceof StructRefDefinition)) {
            return false;
        }
        StructRefDefinition other = (StructRefDefinition) obj;
        return name.equals(other.getName());
    }

    @Override
    public int hashCode() {
        return 17 + 31 * name.hashCode();
    }

    @Override
    public String toString() {
        StringBuffer buf = new StringBuffer();
        buf.append(getName() + " {...}");
        return buf.toString();
    }

}
