/* **********************************************************
 * Copyright 2011-2012 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings;

import static com.vmware.vapi.MessageFactory.getMessage;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.internal.ClassLoaderUtil;
import com.vmware.vapi.provider.ApiMethodBasedApiInterface;

/**
 * Base <code>ApiInterface</code> implementation, intended to be subclassed
 * by API interfaces generated by the vAPI Java generator.
 */
public abstract class ApiInterfaceSkeleton extends ApiMethodBasedApiInterface {
    private final TypeConverter converter;

    /**
     * Constructor.
     *
     * @param iface interface name
     */
    public ApiInterfaceSkeleton(String iface) {
        this(iface, new TypeConverterImpl());
    }

    /**
     * Constructor.
     *
     * @param iface interface name
     * @param converter converter between API runtime types and bindings types
     */
    public ApiInterfaceSkeleton(String iface,
                                TypeConverter converter) {
        super(iface);
        this.converter = converter;
    }

    /**
     * Returns a converter between API runtime types and bindings types.
     *
     * @return converter object
     */
    protected TypeConverter getTypeConverter() {
        return converter;
    }

    protected static final <T> T createImplInstance(Class<? extends T> implClass,
                                                    String defaultImplClass,
                                                    Class<T> ifaceClass) {
        try {
            if (implClass == null) {
                ClassLoader cl = ClassLoaderUtil.getServiceClassLoader();
                // try to load default implementation class
                implClass = cl.loadClass(defaultImplClass).asSubclass(ifaceClass);
            }

            return implClass.newInstance();
        } catch (Exception ex) {
            throw new CoreException(getMessage("vapi.java.invoke.noimpl",
                                               ifaceClass.getCanonicalName()),
                                    ex);
        }
    }

}
