/* **********************************************************
 * Copyright 2013-2015, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings;

import java.util.Arrays;
import java.util.HashSet;
import java.util.Set;

import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.data.ConstraintValidationException;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.data.ConstraintValidator;
import com.vmware.vapi.internal.util.Validate;

/**
 * {@link ConstraintValidator} for <code>@IsOneOf</code> constraint.
 *
 * <p>Each instance is capable of validating a single is-one-of constraint
 * on a single field of a structure.
 */
public class IsOneOfValidator implements ConstraintValidator {

    private final String fieldName;
    private final Set<String> allowedValues;

    /**
     * Constructor.
     *
     * @param fieldName the canonical name of the field that should be validated;
     *                  must not be {@code null}
     * @param allowedValues allowed values for the field; must not be {@code null}
     * @throws IllegalArgumentException if any of the parameters is {@code null}
     */
    public IsOneOfValidator(String fieldName, String[] allowedValues) {
        Validate.notNull(fieldName);
        Validate.notEmpty(allowedValues);
        this.fieldName = fieldName;
        this.allowedValues = new HashSet<>(Arrays.asList(allowedValues));
    }

    /**
     * {@inheritDoc}
     *
     * <p>Constraints:
     * <ul>
     * <li>{@code dataValue} must be a {@code StructValue} instance
     * <li>which contains the expected field with {@code fieldName} of {@code StringValue} type
     * <li>and the value of that field is on of the {@code allowedValues}
     * </ul>
     *
     * @param dataValue {@inheritDoc}
     * @throws ConstraintValidationException {@inheritDoc}
     * @see #IsOneOfValidator
     */
    @Override
    public void validate(DataValue dataValue) {
        /*
         * @IsOneOf is only allowed on String fields, and not on Optional<String>
         * ones (this is enforced by the idl toolkit. So it is safe to assume the
         * field must be present, and the usual versioning requirement to handle
         * missing optional fields doesn't apply here.
         */
        if (!(dataValue instanceof StructValue)) {
            throw new ConstraintValidationException(MessageFactory.getMessage(
                    "vapi.data.structure.isoneof.type.input.mismatch"));
        }

        StructValue struct = (StructValue) dataValue;
        if (!struct.hasField(fieldName)) {
            throw new ConstraintValidationException(MessageFactory.getMessage(
                    "vapi.data.structure.isoneof.missing",
                    struct.getName(),
                    fieldName));
        }

        DataValue fieldValue = struct.getField(fieldName);
        if (!(fieldValue instanceof StringValue)) {
            throw new ConstraintValidationException(MessageFactory.getMessage(
                    "vapi.data.structure.isoneof.type.mismatch", fieldName));
        }

        if (!allowedValues.contains(((StringValue) fieldValue).getValue())) {
            throw new ConstraintValidationException(MessageFactory.getMessage(
                    "vapi.data.structure.isoneof.value.invalid",
                    fieldValue.toString(),
                    fieldName));
        }
    }
}
