/* **********************************************************
 * Copyright 2011-2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings;

import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.OptionalType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;

/**
 * Builds a <code>StructValue</code> instance. Java values for the struct's
 * fields are converted to vAPI <code>DataValue</code> representation and
 * populated to that instance.
 */
public class StructValueBuilder {
    private final StructType structType;
    private final TypeConverter converter;
    private final Map<String, Object> fields = new HashMap<String, Object>();

    /**
     * Constructor.
     *
     * @param structType <code>StructType</code> defining the result type; must
     *                   not be <code>null</code>
     * @param converter converter between API runtime types and bindings types;
     *                  must not be <code>null</code>
     */
    public StructValueBuilder(StructType structType,
                              TypeConverter converter) {
        Validate.notNull(structType);
        Validate.notNull(converter);
        this.structType = structType;
        this.converter = converter;
    }

    /**
     * Adds a value for particular field in the <code>StructValue</code>
     * that is being built.
     *
     * @param name name of the field in the structure
     * @param value value for that field
     */
    public void addStructField(String name, Object value) {
        fields.put(name, value);
    }

    /**
     * @return the built <code>StructValue</code>
     */
    public StructValue getStructValue() {
        StructValue structValue = new StructValue(structType.getName());
        for (String name : structType.getFieldNames()) {
            Type fieldType = structType.getField(name);
            Object fieldBinding = fields.get(name);
            if (fieldBinding == null
                    && !(fieldType instanceof OptionalType)) {
                // missing required operation parameter
                throw new ConverterException(MessageFactory.getMessage(
                        "vapi.bindings.structvaluebuilder.operation.parameter.missing",
                        name));
            }
            try {
                DataValue fieldVal =
                        converter.convertToVapi(fieldBinding, fieldType);
                structValue.setField(name, fieldVal);
            } catch (RuntimeException ex) {
                throw new ConverterException(MessageFactory.getMessage(
                        "vapi.bindings.structvaluebuilder.operation.parameter.error",
                        name), ex);
            }
        }

        return structValue;
    }
}
