/* **********************************************************
 * Copyright 2012-2013 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

/*
 * PrimitiveConverter.java --
 *
 *      Converter for a primitive type
 */
package com.vmware.vapi.internal.bindings.convert;

import com.vmware.vapi.data.ConstraintValidationException;
import com.vmware.vapi.data.DataValue;

/**
 * Converter for a primitive type (e.g. integer, boolean, ...).
 *
 * <p>
 * <i>Thread-safety:</i> Implementations must be thread-safe.
 * </p>
 *
 * @param <T> data value which represents the primitive type in the API runtime
 */
public interface PrimitiveConverter<T extends DataValue> {

    /**
     * Converts an API runtime data value to a binding object. The binding
     * object can be <code>null</code> if it makes sense for the converter
     * implementation.
     *
     * @param value  API runtime data value; must not be null
     * @return binding object
     * @throws ConverterException for general conversion error
     * @throws ConstraintValidationException if conversion can't happen due
     *         to malformed input (e.g. <tt>URI</tt> or <tt>DateTime</tt>
     *         which don't follow the expected string format
     */
    Object fromValue(T value);

    /**
     * Converts a binding object to an API runtime data value. The binding
     * object must be produced by the {@link #fromValue(DataValue)} method
     * of the same converter implementation class. The binding object may
     * be <code>null</code> if it makes sense for the converter implementation.
     *
     * @param primitiveBinding  binding object; must be produced by the same
     *                          converter implementation
     * @return API runtime data value; must not be null
     * @throws ConverterException conversion error
     */
    T toValue(Object primitiveBinding);
}
