/* **********************************************************
 * Copyright (c) 2012-2015, 2017, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings.convert.impl;

import com.vmware.vapi.bindings.ApiError;
import com.vmware.vapi.bindings.Structure;
import com.vmware.vapi.bindings.type.AnyErrorType;
import com.vmware.vapi.bindings.type.DynamicStructType;
import com.vmware.vapi.bindings.type.EnumType;
import com.vmware.vapi.bindings.type.ErrorType;
import com.vmware.vapi.bindings.type.ListType;
import com.vmware.vapi.bindings.type.MapType;
import com.vmware.vapi.bindings.type.OptionalType;
import com.vmware.vapi.bindings.type.SetType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.data.BlobValue;
import com.vmware.vapi.data.BooleanValue;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DoubleValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.data.IntegerValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.SecretValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.data.VoidValue;
import com.vmware.vapi.internal.bindings.convert.ConverterFactory;
import com.vmware.vapi.internal.bindings.convert.NameToTypeResolver;
import com.vmware.vapi.internal.bindings.convert.PrimitiveConverter;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;
import com.vmware.vapi.internal.util.Validate;

/**
 * Default converter factory
 */
public class DefaultConverterFactory implements ConverterFactory {

    private PrimitiveConverter<IntegerValue> integerConverter =
            new JavaLangLongIntegerConverter();

    private PrimitiveConverter<DoubleValue> doubleConverter =
            new JavaLangDoubleDoubleConverter();

    private PrimitiveConverter<BooleanValue> booleanConverter =
            new JavaLangBooleanBooleanConverter();

    private PrimitiveConverter<StringValue> stringConverter =
            new JavaLangStringStringConverter();

    private PrimitiveConverter<BlobValue> binaryConverter =
            new ByteArrayBinaryConverter();

    private PrimitiveConverter<SecretValue> secretConverter =
            new CharArraySecretConverter();

    private PrimitiveConverter<StringValue> dateTimeConverter =
            new JavaUtilCalendarRfc3339DateTimeConverter();

    private PrimitiveConverter<StringValue> uriConverter =
            new JavaNetUriUriConverter();

    private PrimitiveConverter<VoidValue> voidConverter =
            new NullVoidConverter();

    private PrimitiveConverter<DataValue> opaqueConverter =
            new DataValueOpaqueConverter();

    private final UniTypeConverter<StructValue, DynamicStructType> dynamicStructConverter;

    private final UniTypeConverter<ListValue, ListType> listConverter =
            new JavaUtilListListConverter();

    private final UniTypeConverter<StructValue, StructType> structConverter =
            new JavaClassStructConverter<>(StructValue.class, Structure.class);

    private UniTypeConverter<ErrorValue, ErrorType> errorConverter =
            new JavaClassStructConverter<>(ErrorValue.class, ApiError.class);

    private final UniTypeConverter<OptionalValue, OptionalType> optionalConverter =
            new NullableReferenceOptionalConverter();

    private UniTypeConverter<StringValue, EnumType> enumConverter =
            new ApiEnumerationEnumConverter();

    private PrimitiveConverter<StringValue> idConverter =
            new JavaLangStringIdConverter();

    private UniTypeConverter<ListValue, SetType> setConverter =
            new JavaUtilSetSetConverter();

    private final UniTypeConverter<DataValue, MapType> mapConverter =
            new JavaUtilMapMapConverter();

    private final UniTypeConverter<ErrorValue, AnyErrorType> anyErrorConverter;

    /**
     * Creates a converter factory which automatically resolves all standard
     * errors that come through {@link DynamicStructType} and
     * {@link AnyErrorType} fields.
     */
    public DefaultConverterFactory() {
        // TODO: we don't want the core bindings classes to depend on the
        // code-generated bindings;
        // we could either move this dependency injection into the generated
        // code or move it into a new class in a new bindings package which is
        // allowed to depend on the generated bindings; this way we will not
        // have funny dependencies between the bindings packages
        this(new MapBasedNameToTypeResolver(
                MapBasedNameToTypeResolver.STANDARD_ERRORS));
    }

    /**
     * Creates a converter factory which uses the specified resolver to
     * determine the specific type of values that come through
     * {@link DynamicStructType} and {@link AnyErrorType} fields.
     *
     * @param nameToTypeResolver resolver for named types; must not be
     *                           <code>null</code>
     */
    public DefaultConverterFactory(NameToTypeResolver nameToTypeResolver) {
        Validate.notNull(nameToTypeResolver);
        this.anyErrorConverter = new StaticErrorAnyErrorConverter(
                nameToTypeResolver);
        this.dynamicStructConverter = new StaticStructureDynamicStructureConverter(
                nameToTypeResolver);
    }

    @Override
    public PrimitiveConverter<IntegerValue> getIntegerConverter() {
        return integerConverter;
    }

    @Override
    public PrimitiveConverter<DoubleValue> getDoubleConverter() {
        return doubleConverter;
    }

    @Override
    public PrimitiveConverter<BooleanValue> getBooleanConverter() {
        return booleanConverter;
    }

    @Override
    public PrimitiveConverter<StringValue> getStringConverter() {
        return stringConverter;
    }

    @Override
    public PrimitiveConverter<BlobValue> getBinaryConverter() {
        return binaryConverter;
    }

    @Override
    public PrimitiveConverter<SecretValue> getSecretConverter() {
        return secretConverter;
    }

    @Override
    public PrimitiveConverter<StringValue> getDateTimeConverter() {
        return dateTimeConverter;
    }

    @Override
    public PrimitiveConverter<StringValue> getUriConverter() {
        return uriConverter;
    }

    @Override
    public PrimitiveConverter<VoidValue> getVoidConverter() {
        return voidConverter;
    }

    @Override
    public PrimitiveConverter<DataValue> getOpaqueConverter() {
        return opaqueConverter;
    }

    @Override
    public UniTypeConverter<StructValue, DynamicStructType> getDynamicStructureConverter() {
        return dynamicStructConverter;
    }

    @Override
    public UniTypeConverter<ListValue, ListType> getListConverter() {
        return listConverter;
    }

    @Override
    public UniTypeConverter<StructValue, StructType> getStructConverter() {
        return structConverter;
    }

    @Override
    public UniTypeConverter<ErrorValue, ErrorType> getErrorConverter() {
        return errorConverter;
    }

    @Override
    public UniTypeConverter<OptionalValue, OptionalType> getOptionalConverter() {
        return optionalConverter;
    }

    @Override
    public UniTypeConverter<StringValue, EnumType> getEnumConverter() {
        return enumConverter;
    }

    @Override
    public PrimitiveConverter<StringValue> getIdConverter() {
        return idConverter ;
    }

    @Override
    public UniTypeConverter<ListValue, SetType> getSetConverter() {
        return setConverter;
    }

    @Override
    public UniTypeConverter<DataValue, MapType> getMapConverter() {
        return mapConverter;
    }

    @Override
    public UniTypeConverter<ErrorValue, AnyErrorType> getAnyErrorConverter() {
        return anyErrorConverter;
    }

}
