/* **********************************************************
 * Copyright (c) 2017-2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings.convert.impl;

import java.text.MessageFormat;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.bindings.type.EnumType;
import com.vmware.vapi.bindings.type.IdType;
import com.vmware.vapi.bindings.type.MapType;
import com.vmware.vapi.bindings.type.StringType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.bindings.type.UriType;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverter.ConversionContext;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;
import com.vmware.vapi.std.BuiltInDataFactory;

/**
 * Converts a {@code DataValue} represented map to a {@code java.util.Map}.
 *
 * <p>The actual implementation used for the native map type is {@link HashMap}.
 *
 * <p>The map is assumed to be represented as a {@code StructValue} instance,
 * which field names are the keys for the map, and field values are the
 * respective values for the map. Thus, the map keys are always of type
 * {@code String}.
 */
public class JavaUtilMapStructValueMapConverter implements
        UniTypeConverter<DataValue, MapType> {

    private static final String MAP_STRUCT_NAME =
                            BuiltInDataFactory.MAP_STRUCT_NAME;
    private static final String UNEXPECTED_TYPE_MSG =
                            "vapi.bindings.typeconverter.unexpected.type";

    @Override
    public Object fromValue(DataValue value,
                            MapType declaredType,
                            TypeConverter typeConverter) {
        return convertStructValueToMap(value, declaredType, typeConverter);
    }

    @Override
    public DataValue toValue(Object binding,
                             MapType declaredType,
                             TypeConverter typeConverter,
                             ConversionContext cc) {
        @SuppressWarnings("unchecked")
        Map<Object, Object> mapBinding =
                         ConvertUtil.narrowType(binding, Map.class);
        ensureStringKeyType(declaredType.getKeyType());

        Type valueType = declaredType.getValueType();
        StructValue mapValue = new StructValue(MAP_STRUCT_NAME);
        for (Map.Entry<Object, Object> elem : mapBinding.entrySet()) {
            String key = ConvertUtil.narrowType(elem.getKey(), String.class);
            Object value = elem.getValue();
            DataValue val = typeConverter.convertToVapi(value, valueType, cc);
            mapValue.setField(key , val);
        }

        return mapValue;
    }

    static Object convertStructValueToMap(DataValue value,
                                          MapType declaredType,
                                          TypeConverter typeConverter) {
        StructValue structValue = ConvertUtil.narrowType(value,
                                                         StructValue.class);
        ensureStringKeyType(declaredType.getKeyType());

        Type valueType = declaredType.getValueType();
        Map<Object, Object> mapBinding = new HashMap<>();

        for (String fieldName : structValue.getFieldNames()) {
            DataValue field = structValue.getField(fieldName);
            Object val = typeConverter.convertToJava(field, valueType);
            mapBinding.put(fieldName, val);
        }

        return mapBinding;
    }

    private static final List<Class<? extends Type>> supportedKeyTypes =
            Arrays.asList(StringType.class,
                          IdType.class,
                          EnumType.class,
                          UriType.class);
    private static final String supportedTypesNames = MessageFormat
            .format("{0}, {1}, {2} or {3}",
                    StringType.class.getSimpleName(),
                    IdType.class.getSimpleName(),
                    EnumType.class.getSimpleName(),
                    UriType.class.getSimpleName());

    private static void ensureStringKeyType(Type keyType) {
        if (supportedKeyTypes.contains(keyType.getClass())) {
            return;
        }
        throw new ConverterException(UNEXPECTED_TYPE_MSG,
                                     supportedTypesNames,
                                     keyType.getClass().getSimpleName());

    }
}
