/* **********************************************************
 * Copyright (c) 2012-2013, 2016, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings.convert.impl;

import com.vmware.vapi.bindings.type.OptionalType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverter.ConversionContext;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;

/**
 * Convert optional type to nullable object reference.
 */
public final class NullableReferenceOptionalConverter implements
        UniTypeConverter<OptionalValue, OptionalType> {

    @Override
    public Object fromValue(OptionalValue value, OptionalType declaredType,
            TypeConverter typeConverter) {
        Object element = null;
        if (value != null && value.isSet()) {
            Type elementType = declaredType.getElementType();
            DataValue elementVal = value.getValue();
            element = typeConverter.convertToJava(elementVal, elementType);
        } else if (declaredType.getElementType() instanceof OptionalType) {
            /*
             * If the OptionalValue is not set, but the optional types are
             * nested, still build (empty) element bindings to produce the
             * maximum possible nesting.
             */
            Type elementType = declaredType.getElementType();
            // need an empty OptionalValue, so reuse the input value
            element = typeConverter.convertToJava(value, elementType);
        }
        return element;
    }

    @Override
    public DataValue toValue(Object binding,
                             OptionalType declaredType,
                             TypeConverter typeConverter,
                             ConversionContext cc) {
        /*
         * If the element binding is not set, but the optional types are
         * nested, still build OptionalValues to produce the maximum
         * possible nesting.
         */
        if (binding != null
            || declaredType.getElementType() instanceof OptionalType) {
            DataValue e = typeConverter
                    .convertToVapi(binding, declaredType.getElementType(), cc);
            return new OptionalValue(e);
        } else {
            return OptionalValue.UNSET;
        }
    }

}
