/* **********************************************************
 * Copyright (c) 2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.bindings.convert.impl;

import com.vmware.vapi.bindings.StaticStructure;
import com.vmware.vapi.bindings.UnresolvedError;
import com.vmware.vapi.bindings.type.AnyErrorType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.internal.bindings.TypeConverter;
import com.vmware.vapi.internal.bindings.TypeConverter.ConversionContext;
import com.vmware.vapi.internal.bindings.convert.NameToTypeResolver;
import com.vmware.vapi.internal.bindings.convert.UniTypeConverter;
import com.vmware.vapi.internal.util.Validate;

/**
 * Converter for {@link AnyErrorType} which uses a {@link NameToTypeResolver} to
 * produce specific bindings for known error types.
 */
public final class StaticErrorAnyErrorConverter implements
        UniTypeConverter<ErrorValue, AnyErrorType> {

    private final NameToTypeResolver resolver;

    /**
     * Creates a converter which uses the specified resolver.
     *
     * @param resolver name resolution policy; must not be <code>null</code>
     */
    public StaticErrorAnyErrorConverter(NameToTypeResolver resolver) {
        Validate.notNull(resolver);
        this.resolver = resolver;
    }

    @Override
    public Object fromValue(ErrorValue value,
                            AnyErrorType declaredType,
                            TypeConverter typeConverter) {
        Type type = resolver.resolve(value.getName());
        if (type != null) {
            return typeConverter.convertToJava(value, type);
        }
        return new UnresolvedError(value);
    }

    @Override
    public DataValue toValue(Object binding,
                             AnyErrorType declaredType,
                             TypeConverter typeConverter,
                             ConversionContext cc) {
        if (binding instanceof StaticStructure) {
            StaticStructure struct = (StaticStructure) binding;
            Type type = struct._getType();
            return typeConverter.convertToVapi(binding, type, cc);
        }
        UnresolvedError error = ConvertUtil.narrowType(binding,
                UnresolvedError.class);
        return error._getDataValue();
    }

}
