/* **********************************************************
 * Copyright 2012-2015, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.bindings.type;

import static com.vmware.vapi.internal.bindings.convert.Constants.MAP_TYPE_ENTRY_KEY_FIELD;
import static com.vmware.vapi.internal.bindings.convert.Constants.MAP_TYPE_ENTRY_VALUE_FIELD;

import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.vmware.vapi.bindings.type.AnyErrorType;
import com.vmware.vapi.bindings.type.BinaryType;
import com.vmware.vapi.bindings.type.BooleanType;
import com.vmware.vapi.bindings.type.DateTimeType;
import com.vmware.vapi.bindings.type.DoubleType;
import com.vmware.vapi.bindings.type.DynamicStructType;
import com.vmware.vapi.bindings.type.EnumType;
import com.vmware.vapi.bindings.type.ErrorType;
import com.vmware.vapi.bindings.type.IdType;
import com.vmware.vapi.bindings.type.IntegerType;
import com.vmware.vapi.bindings.type.ListType;
import com.vmware.vapi.bindings.type.MapType;
import com.vmware.vapi.bindings.type.OpaqueType;
import com.vmware.vapi.bindings.type.OptionalType;
import com.vmware.vapi.bindings.type.SecretType;
import com.vmware.vapi.bindings.type.SetType;
import com.vmware.vapi.bindings.type.StringType;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.bindings.type.Type;
import com.vmware.vapi.bindings.type.TypeReference;
import com.vmware.vapi.bindings.type.TypeVisitor;
import com.vmware.vapi.bindings.type.UriType;
import com.vmware.vapi.bindings.type.VoidType;
import com.vmware.vapi.data.AnyErrorDefinition;
import com.vmware.vapi.data.BlobDefinition;
import com.vmware.vapi.data.BooleanDefinition;
import com.vmware.vapi.data.DataDefinition;
import com.vmware.vapi.data.DoubleDefinition;
import com.vmware.vapi.data.DynamicStructDefinition;
import com.vmware.vapi.data.ErrorDefinition;
import com.vmware.vapi.data.IntegerDefinition;
import com.vmware.vapi.data.ListDefinition;
import com.vmware.vapi.data.OpaqueDefinition;
import com.vmware.vapi.data.OptionalDefinition;
import com.vmware.vapi.data.SecretDefinition;
import com.vmware.vapi.data.StringDefinition;
import com.vmware.vapi.data.StructDefinition;
import com.vmware.vapi.data.StructRefDefinition;
import com.vmware.vapi.data.VoidDefinition;
import com.vmware.vapi.internal.data.ReferenceResolver;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.std.BuiltInDataFactory;

/**
 * Operations on bindings' type descriptors.
 */
public final class TypeUtil {

    private TypeUtil() {
    }

    /**
     * Convert this type representation to the corresponding type representation
     * in the API runtime.
     *
     * @param type bindings' type descriptor; must not be <code>null</code>
     * @return type representation in the API runtime
     */
    public static DataDefinition toDataDefinition(Type type) {
        Validate.notNull(type);
        ReferenceResolver ctx = new ReferenceResolver();
        Set<String> seenStructures = new HashSet<String>();
        DataDefinition def = DataDefinitionBuilder.toDataDefinition(
                type, ctx, seenStructures);
        ctx.resolveReferences();
        return def;
    }

    /**
     * Type visitor which builds a corresponding {@link DataDefinition} for a
     * type. In the resulting definition, structure references are left
     * unresolved.
     */
    private static class DataDefinitionBuilder implements TypeVisitor {

        private final ReferenceResolver ctx;

        private final Set<String> seenStructures;

        private DataDefinition def;

        public DataDefinitionBuilder(ReferenceResolver ctx,
                                     Set<String> seenStructures) {
            this.ctx = ctx;
            this.seenStructures = seenStructures;
        }

        public DataDefinition getDefinition() {
            return def;
        }

        @Override
        public void visit(VoidType type) {
            def = VoidDefinition.getInstance();
        }

        @Override
        public void visit(IntegerType type) {
            def = IntegerDefinition.getInstance();
        }

        @Override
        public void visit(DoubleType type) {
            def = DoubleDefinition.getInstance();
        }

        @Override
        public void visit(StringType type) {
            def = StringDefinition.getInstance();
        }

        @Override
        public void visit(BooleanType type) {
            def = BooleanDefinition.getInstance();
        }

        @Override
        public void visit(BinaryType type) {
            def = BlobDefinition.getInstance();
        }

        @Override
        public void visit(DateTimeType type) {
            def = StringDefinition.getInstance();
        }

        @Override
        public void visit(UriType type) {
            def = StringDefinition.getInstance();
        }

        @Override
        public void visit(OptionalType type) {
            def = new OptionalDefinition(
                    toDataDefinition(type.getElementType()));
        }

        @Override
        public void visit(ListType type) {
            def = new ListDefinition(
                    toDataDefinition(type.getElementType()));
        }

        @Override
        public void visit(StructType type) {
            String structName = type.getName();
            if (seenStructures.contains(structName)) {
                if(ctx.isDefined(structName)){
                    def = ctx.getDefinition(structName);
                } else {
                    // For recursive/self-referencing structures the resolution
                    // of first/top-level is not yet completed when we visit the
                    // second level of structure reference.
                    // So, to break the cycle in building definitions we create
                    // an unresolved reference (StructRefDefinition) to end it.
                    StructRefDefinition structRef =
                            new StructRefDefinition(structName);
                    ctx.addReference(structRef);
                    def = structRef;
                }
            } else {
                seenStructures.add(structName);
                Map<String, DataDefinition> fields =
                        new HashMap<String, DataDefinition>();
                for (String field : type.getFieldNames()) {
                    Type fieldType = type.getField(field);
                    DataDefinition fieldDef = toDataDefinition(fieldType);
                    fields.put(field, fieldDef);
                }
                StructDefinition structDef =
                        new StructDefinition(structName, fields);
                ctx.addDefinition(structDef);
                def = structDef;
            }
        }

        @Override
        public void visit(OpaqueType type) {
            def = OpaqueDefinition.getInstance();
        }

        @Override
        public void visit(SecretType type) {
            def = SecretDefinition.getInstance();
        }

        @Override
        public void visit(TypeReference<? extends Type> type) {
            Type target = type.resolve();
            target.accept(this);
        }

        @Override
        public void visit(EnumType type) {
            def = StringDefinition.getInstance();
        }

        @Override
        public void visit(ErrorType type) {
            // TODO: handle circular references between errors the same way as
            // we handle structs; see comments in StructRefDefinition
            Map<String, DataDefinition> fields =
                    new HashMap<String, DataDefinition>();
            for (String field : type.getFieldNames()) {
                Type fieldType = type.getField(field);
                DataDefinition fieldDef = toDataDefinition(fieldType);
                fields.put(field, fieldDef);
            }
            def = new ErrorDefinition(type.getName(), fields);
        }

        @Override
        public void visit(SetType type) {
            def = new ListDefinition(
                    toDataDefinition(type.getElementType()));
        }

        @Override
        public void visit(MapType type) {
            Map<String, DataDefinition> fields =
                    new HashMap<String, DataDefinition>();
            fields.put(MAP_TYPE_ENTRY_KEY_FIELD,
                       toDataDefinition(type.getKeyType()));
            fields.put(MAP_TYPE_ENTRY_VALUE_FIELD,
                       toDataDefinition(type.getValueType()));
            def = new ListDefinition(
                    new StructDefinition(
                            BuiltInDataFactory.MAP_ENTRY_STRUCT_NAME,
                            fields));
        }

        @Override
        public void visit(DynamicStructType type) {
            def = DynamicStructDefinition.getInstance();
        }

        @Override
        public void visit(IdType idType) {
            def = StringDefinition.getInstance();
        }

        @Override
        public void visit(AnyErrorType type) {
            def = AnyErrorDefinition.getInstance();
        }

        private DataDefinition toDataDefinition(Type type) {
            return toDataDefinition(type, ctx, seenStructures);
        }

        public static DataDefinition toDataDefinition(
                Type type,
                ReferenceResolver ctx,
                Set<String> seenStructures) {
            DataDefinitionBuilder elementDefBuilder =
                    new DataDefinitionBuilder(ctx, seenStructures);
            type.accept(elementDefBuilder);
            return elementDefBuilder.getDefinition();
        }
    }
}