/* **********************************************************
 * Copyright (c) 2015-2020, 2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.client;

import java.io.Closeable;
import java.io.IOException;

import org.apache.http.client.config.RequestConfig;

import com.vmware.vapi.bindings.Service;
import com.vmware.vapi.bindings.StubConfiguration;
import com.vmware.vapi.bindings.StubConfigurationBase;
import com.vmware.vapi.bindings.StubFactory;
import com.vmware.vapi.client.ApiClient;
import com.vmware.vapi.client.Configuration;
import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.internal.client.Protocol.ApiProviderCardinality;
import com.vmware.vapi.internal.protocol.client.rpc.http.ApacheBioHttpClientBuilder;
import com.vmware.vapi.internal.protocol.client.rpc.http.ApacheNioHttpClientBuilder;
import com.vmware.vapi.internal.protocol.client.rpc.http.ConnectionMonitor.CleanableConnectionPool;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration;

/**
 * SDK Client for APIs.
 */
public final class ApiClientImpl implements ApiClient {
    private final Configuration config;
    private final String url;
    private final Protocol protocol;
    private final StubConfigurationBase stubConfig;

    // shared underlying transport (e.g. Apache BIO/NIO HTTP client) to be reused
    // by all stubs in use in this client
    private Closeable httpClient;
    private String httpClientMonicker;
    private RequestConfig defaultRequestConfig;
    private CleanableConnectionPool pool;


    // shared ApiProvider to be reused by all stubs created by this ApiClient,
    // only if Protocol supports this
    private final ApiProvider apiProvider;

    /**
     * Constructor.
     *
     * <p>Creates instance with the specified configuration for the <tt>HTTP</tt> stack.
     *
     * @param url location of the API to be consumed; must not be {@code null}
     * @param protocol protocol to use for invoking the API; must not be {@code null}
     * @param config additional configuration for this client instance; must
     *               not be {@code null}
     */
    public ApiClientImpl(String url,
                         Protocol protocol,
                         Configuration config) {
        Validate.notNull(url);
        Validate.notNull(protocol);
        Validate.notNull(config);
        this.url = url;
        this.config = config;

        this.protocol = protocol;
        this.stubConfig = config.getProperty(Configuration.STUB_CONFIG_CFG,
                                             StubConfiguration.class);

        prepareSharedTransport(config);
        this.apiProvider = prepareSharedApiProvider();
    }

    /**
     * Creates a remote stub for the specified interface and <i>API endpoint</i> base URL.
     *
     * @param vapiIface <code>Class</code> a vAPI service interface; must
     *                  not be <code>null</code>
     *
     * @return a stub instance for the specified vAPI interface
     */
    @Override
    public <T extends Service> T createStub(Class<T> vapiIface) {
        return createStub(vapiIface, null);
    }

    /**
     * Creates a remote stub for the specified interface and <i>API endpoint</i> base URL.
     *
     * @param vapiIface <code>Class</code> a vAPI service interface; must
     *                  not be <code>null</code>
     * @param stubConfiguration the stub's additional configuration
     * @return a stub instance for the specified vAPI interface
     */
    @Override
    public <T extends Service> T createStub(Class<T> vapiIface,
                                            StubConfigurationBase stubConfiguration) {
        StubConfigurationBase effectiveStubConfig =
                stubConfiguration != null ? stubConfiguration
                                          : this.stubConfig;

        StubFactory stubFactory = new StubFactory(findRestApiProvider(vapiIface),
                                                  protocol.getTypeConverter(effectiveStubConfig, vapiIface));
        return stubFactory.createStub(vapiIface, effectiveStubConfig);
    }

    @Override
    public void close() throws IOException {
        if (httpClient != null) {
            httpClient.close();
        }
        // TODO: need to make sure the contract of Transport/Connection in case
        //       a.Client is injected to them is that they do not close but expect
        //       consumer to close
    }

    final ApiProvider prepareSharedApiProvider() {
        // TODO: rest native: support one configured in Configuration
        if (protocol.getApiProviderCardinality() == ApiProviderCardinality.SINGLE) {
            return buildApiProvider(null);
        }
        return null;
    }

    final void prepareSharedTransport(Configuration config) {
        HttpConfiguration httpConfig = config.getProperty(Configuration.HTTP_CONFIG_CFG,
                                                          HttpConfiguration.class);
        if (httpConfig == null) {
            // now and in the near future we only support HTTP transports,
            // so let's use default HTTP config here
            httpConfig = new HttpConfiguration.Builder().getConfig();
        }

        switch (httpConfig.getLibraryType()) {
            case APACHE_HTTP_ASYNC_CLIENT:
                httpClientMonicker = Protocol.NIO_HTTP_CLIENT_MONICKER;
                ApacheNioHttpClientBuilder httpAsyncClientBuilder = new ApacheNioHttpClientBuilder();
                httpClient = httpAsyncClientBuilder.buildAndConfigure(httpConfig);
                defaultRequestConfig = httpAsyncClientBuilder.getDefaultRequestConfig();
                pool = httpAsyncClientBuilder.registerClientWithConnectionMonitor();
                break;

            case APACHE_HTTP_CLIENT:
                httpClientMonicker = Protocol.BIO_HTTP_CLIENT_MONICKER;
                ApacheBioHttpClientBuilder bioHttpClientBuilder = new ApacheBioHttpClientBuilder();
                httpClient = bioHttpClientBuilder.buildAndConfigure(httpConfig);
                defaultRequestConfig = bioHttpClientBuilder.getDefaultRequestConfig();
                pool = bioHttpClientBuilder.registerClientWithConnectionMonitor();
                break;
            default:
                throw new RuntimeException("Unsupported library type: " +
                        httpConfig.getLibraryType());
        }
    }

    ApiProvider findRestApiProvider(Class<? extends Service> vapiIface) {
        if (apiProvider != null) {
            return apiProvider;
        }

        return buildApiProvider(vapiIface);
    }

    ApiProvider buildApiProvider(Class<? extends Service> vapiIface) {
        Configuration.Builder enhancedConfig = new Configuration.Builder(config);
        if (httpClient != null) {
            enhancedConfig.register(httpClientMonicker, httpClient);
        }
        if (defaultRequestConfig != null) {
            enhancedConfig.register(Protocol.DEFAULT_REQUEST_CONFIG_MONICKER,
                                    defaultRequestConfig);
        }
        if (vapiIface != null) {
            enhancedConfig.register(Protocol.VAPI_INTERFACE_MONICKER,
                                    vapiIface);
        }
        if (pool != null) {
            enhancedConfig.register(Protocol.VAPI_CONNECTION_POOL,
                                    pool);
        }
        return protocol.getProtocolConnection(url, enhancedConfig.build()).getApiProvider();
    }
}
