/* **********************************************************
 * Copyright 2014, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.core.abort;

import java.util.ArrayList;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.util.Validate;

/**
 * Default implementation for the {@link AbortHandle} interface.
 */
public class AbortHandleImpl implements AbortHandle {
    private static final Logger LOGGER = LoggerFactory
            .getLogger(AbortHandleImpl.class);

    private boolean isAborted = false;
    private final List<AbortListener> listeners = new ArrayList<AbortListener>();

    @Override
    public synchronized void addAbortListener(AbortListener listener) {
        Validate.notNull(listener);
        if (isAborted()) {
            return;
        }
        listeners.add(listener);
    }

    @Override
    public void abort() {
        synchronized (this) {
            if (isAborted) {
                return;
            }
            isAborted = true;
        }

        for (AbortListener listener : listeners) {
            try {
                listener.onAbort();
            } catch (RuntimeException e) {
                if (LOGGER.isWarnEnabled()) {
                    String message = String.format("Aborting %s failed",
                                                   listener);
                    LOGGER.warn(message, e);
                }
            }
        }
    }

    @Override
    public synchronized boolean isAborted() {
        return isAborted;
    }

    protected List<AbortListener> getListeners() {
        return listeners;
    }
}