/* **********************************************************
 * Copyright 2013-2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import javax.servlet.Servlet;

import org.apache.catalina.Executor;
import org.apache.catalina.core.StandardThreadExecutor;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.ProtocolHandler;
import com.vmware.vapi.protocol.ProtocolHandlerFactory;
import com.vmware.vapi.protocol.RequestProcessor;
import com.vmware.vapi.protocol.server.msg.json.JsonServerConnection;
import com.vmware.vapi.protocol.server.rpc.http.Endpoint;
import com.vmware.vapi.protocol.server.rpc.http.Server;
import com.vmware.vapi.protocol.server.rpc.http.Service;
import com.vmware.vapi.protocol.server.rpc.http.impl.HttpEndpoint;
import com.vmware.vapi.protocol.server.rpc.http.impl.HttpServer;
import com.vmware.vapi.protocol.server.rpc.http.impl.HttpStreamingServlet;
import com.vmware.vapi.protocol.server.rpc.http.impl.HttpsEndpoint;
import com.vmware.vapi.protocol.server.rpc.http.impl.MediaTypeResolverImpl;
import com.vmware.vapi.protocol.server.rpc.http.impl.ServiceImpl;
import com.vmware.vapi.protocol.server.rpc.http.impl.TcServer;

/**
 * Factory for creating {@link ProtocolHandler} objects. Should be used only for
 * prototypes and tests. For production ready systems wire the server objects
 * directly (e.g. via Spring XML).
 */
public class ProtocolHandlerFactoryImpl implements ProtocolHandlerFactory {

    // TODO find a better place for this enum
    public enum Protocol {
        http
    }

    /**
     * Client authentication type
     */
    public enum ClientAuthentication {
        /**
         * Require a valid certificate from the client to accept a connection
         * and fail if the certificate is not supplied
         */
        need("true"),

        /**
         * Request the client to supply a valid certificate, but do not fail if
         * is not provided
         */
        want("want"),

        /**
         * Do not require a client certificate
         */
        nothing("false");

        String value;
        ClientAuthentication(String value) {
            this.value = value;
        }
    }

    @Override
    public ProtocolHandler getProtocolHandler(String type,
                                              String args,
                                              String secureArgs,
                                              String keystore,
                                              String keystorePassword,
                                              String keyPassword,
                                              ApiProvider provider) {
        if (type.equals(Protocol.http.toString())) {
            return createHttpHandler(args,
                                     secureArgs,
                                     ClientAuthentication.nothing.value,
                                     new KeyStoreConfig(keystore, keystorePassword, keyPassword),
                                     null,
                                     provider);
        } else {
            return null;
        }
    }

    @Override
    public ProtocolHandler getProtocolHandler(String type,
                                              String args,
                                              String secureArgs,
                                              String clientAuth,
                                              KeyStoreConfig keystoreConfig,
                                              TrustStoreConfig truststoreConfig,
                                              ApiProvider provider) {
        if (type.equals(Protocol.http.toString())) {
            return createHttpHandler(args,
                    secureArgs,
                    clientAuth,
                    keystoreConfig,
                    truststoreConfig,
                    provider);
        } else {
            return null;
        }
    }

    private static ProtocolHandler createHttpHandler(String httpUri,
                                                     String httpsUri,
                                                     String clientAuth,
                                                     KeyStoreConfig keystoreConfig,
                                                     TrustStoreConfig truststoreConfig,
                                                     ApiProvider provider) {
        Server server = createHttpServer();
        Servlet servlet = createServlet(provider);
        List<Endpoint> endpoints = new ArrayList<Endpoint>();
        List<Service> services = new ArrayList<Service>();
        if (httpUri != null) {
            URL httpUrl = parseUrl(httpUri);
            endpoints.add(createHttpEndpoint(httpUrl));
            services.add(createService(httpUrl, servlet));
        }
        if (httpsUri != null) {
            Validate.notNull(keystoreConfig);
            Validate.notNull(keystoreConfig.getPath());
            URL httpsUrl = parseUrl(httpsUri);
            endpoints.add(createHttpsEndpoint(httpsUrl, keystoreConfig,
                    truststoreConfig, clientAuth));
            services.add(createService(httpsUrl, servlet));
        }
        server.setEndpoints(endpoints.toArray(new Endpoint[endpoints.size()]));
        server.setServices(services.toArray(new Service[services.size()]));
        return new HttpServer(server);
    }

    private static Servlet createServlet(ApiProvider provider) {
        return new HttpStreamingServlet(
                MediaTypeResolverImpl.newJsonResolver(new JsonServerConnection(
                        provider, Collections.<RequestProcessor>emptyList())));
    }

    private static Server createHttpServer() {
        TcServer server = new TcServer();
        server.setThreadPool(createThreadPool());
        return server;
    }

    private static Executor createThreadPool() {
        StandardThreadExecutor threadPool = new StandardThreadExecutor();
        threadPool.setName("HttpThreadPool");
        threadPool.setNamePrefix("Test-vAPI-tcserver-");
        threadPool.setMinSpareThreads(10);
        threadPool.setMaxThreads(300);
        threadPool.setMaxIdleTime(60000);
        threadPool.setMaxQueueSize(50);
        threadPool.setPrestartminSpareThreads(true);
        threadPool.setDaemon(false);
        return threadPool;
    }

    private static Service createService(URL url, Servlet servlet) {
        ServiceImpl service = new ServiceImpl();
        service.setPath(url.getPath());
        service.setServlet(servlet);
        return service;
    }

    private static HttpEndpoint createHttpEndpoint(URL url) {
        HttpEndpoint endpoint = new HttpEndpoint(url.getPort());
        endpoint.setMaxIdleTime(200000);
        return endpoint;
    }

    private static HttpsEndpoint createHttpsEndpoint(URL url,
                                                     KeyStoreConfig keystoreConfig) {
        HttpsEndpoint endpoint = new HttpsEndpoint(url.getPort());
        if (keystoreConfig != null) {
            endpoint.setKeyStorePath(keystoreConfig.getPath());
            endpoint.setKeyStorePassword(keystoreConfig.getPassword());
            endpoint.setKeyPassword(keystoreConfig.getKeyPassword());
        }
        endpoint.setKeyStoreType("jks");
        endpoint.setMaxIdleTime(200000);
        return endpoint;
    }

    private static HttpsEndpoint createHttpsEndpoint(URL url,
                                                     KeyStoreConfig keystoreConfig,
                                                     TrustStoreConfig truststoreConfig,
                                                     String clientAuth) {
        HttpsEndpoint endpoint = createHttpsEndpoint(url,
                                                     keystoreConfig);
        if (truststoreConfig != null) {
            endpoint.setTrustStorePath(truststoreConfig.getPath());
            endpoint.setTrustStorePassword(truststoreConfig.getPassword());
        }
        if (ClientAuthentication.need.value.equals(clientAuth)) {
            Validate.notNull(truststoreConfig);
            Validate.notNull(truststoreConfig.getPath());
            endpoint.setNeedClientAuth(true);
        } else if (ClientAuthentication.want.value.equals(clientAuth)) {
            endpoint.setWantClientAuth(true);
        }
        return endpoint;
    }

    private static URL parseUrl(String url) {
        try {
            return new URL(url);
        } catch (MalformedURLException ex) {
            throw new IllegalArgumentException("Invalid URL '"+url+"'", ex);
        }
    }

    @Override
    public ProtocolHandler getInsecureProtocolHandler(String type,
                                                      String args,
                                                      ApiProvider provider) {
        if (type.equals(Protocol.http.toString())) {
            return createHttpHandler(args, null, null,
                    null, null, provider);
        } else {
            return null;
        }
    }
}
