/* **********************************************************
 * Copyright (c) 2017 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rest.authn;

import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.internal.protocol.client.rpc.HttpRequest;
import com.vmware.vapi.protocol.client.http.RequestPreProcessor;
import com.vmware.vapi.security.OAuthSecurityContext;
import com.vmware.vapi.security.StdSecuritySchemes;

/**
 * Abstract base class for authentication appenders which use HTTP header
 * in the request to transport an authorization token.
 *
 * <p>The authorization token is extracted from the {@code SecurityContext},
 * only if it uses {@link StdSecuritySchemes#OAUTH} security scheme. Otherwise,
 * no token can be obtained and thus no HTTP header will be added to the
 * request.
 *
 * <p>If the header exists, it will be overwritten.
 */
public abstract class HeaderAuthenticationAppenderBase
        implements RequestPreProcessor, HttpRequestAuthorizer {
    private static final Logger logger =
            LoggerFactory.getLogger(HeaderAuthenticationAppenderBase.class);

    @Override
    public HttpRequest handle(String serviceId,
                              String operationId,
                              HttpRequest request,
                              DataValue params,
                              ExecutionContext context) {
        SecurityContext securityContext = context.retrieveSecurityContext();
        authorize(request, securityContext);
        return request;
    }

    @Override
    public void authorize(HttpRequest httpRequest,
                          SecurityContext securityContext) {
        if (securityContext == null) {
            logger.debug("Authentication header will not be added.");
            return;
        }

        Object authnSchemeId =
                securityContext.getProperty(SecurityContext.AUTHENTICATION_SCHEME_ID);
        logger.debug("Authentication scheme is {}.", authnSchemeId);
        if (StdSecuritySchemes.OAUTH.equals(authnSchemeId)) {
            Object token = securityContext.getProperty(OAuthSecurityContext.ACCESS_TOKEN);
            if (token instanceof char[]) {
                putAuthorizationHeader(httpRequest, new String((char[]) token));
            } else if (token == null) {
                logger.info("No OAuth token found in SecurityContext");
            } else {
                logger.warn("OAuth token in SecurityContext is corrupted");
            }

//      TODO: Cleaner way to implement the previous 2 lines is to use the SCF as below.
//            However it is in vapi-authn module, and adding dependency to this here in
//            vapi-rest is problematic because of the vapi-auth vs vapi-authentication.
//            More precisely adding the dep here breaks vapi-rest-authentication, because
//            it gets both vapi-auth and vapi-authentication in its transitive closure of
//            dependencies. We have at least 2 options:
//               - move this class in vapi-rest-authn
//               - move the SCF logic we need here from vapi-authn to vapi-runtime (be careful
//                 not to move SAML token stuff, they must stay in vapi-authn; so it is not
//                 just a straight class movement)
//
//            OAuthSecurityContext oauthSecCtx =
//                    SecurityContextFactory.parseOAuthSecurityContext(securityContext);
//            addOauthBearerToken(request, new String(oauthSecCtx.getAccessToken()));
        } else {
            logger.warn("No authentication added to request - authentication scheme {} is not supported.",
                        authnSchemeId);
        }
    }

    /**
     * Overrides the authorization header of the HTTP {@code request} with the
     * specified {@code token} as a value. Appends such a header if it doesn't
     * exist.
     *
     * @param request HTTP request
     * @param token the authentication token
     */
    protected void putAuthorizationHeader(HttpRequest request, String token) {
        Objects.requireNonNull(request);
        Objects.requireNonNull(token);
        String name =Objects.requireNonNull(getHeaderName());
        String value = Objects.requireNonNull(getHeaderValue(token));
        Map<String, List<String>> headers = new HashMap<>(request.getHeaders());
        headers.put(name, Arrays.asList(value));
        request.setHeaders(headers);
    }

    /**
     * @return name of the authentication header; never {@code null}
     */
    protected abstract String getHeaderName();

    /**
     * @param token to be transformed into a header value
     * @return the value of the authorization header; never {@code null}
     */
    protected abstract String getHeaderValue(String token);
}
