/* **********************************************************
 * Copyright 2013, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc;

import java.io.InputStream;
import java.util.concurrent.Executor;

import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallback;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.TransportControl;
import com.vmware.vapi.internal.util.Validate;

/**
 * Decorator which allows to run ResponseCallback methods asynchronously.
 */
public final class ExecutorResponseCallback implements ResponseCallback {
    private final ResponseCallback cb;
    private final Executor executor;

    /**
     * Creates a decorator around an existing callback.
     *
     * @param cb the callback to execute; must not be <code>null</code>
     * @param executor executor used to execute the callback methods;
     *                 must not be <code>null</code>
     */
    public ExecutorResponseCallback(ResponseCallback cb, Executor executor) {
        Validate.notNull(cb);
        Validate.notNull(executor);
        this.cb = cb;
        this.executor = executor;
    }

    @Override
    public void received(final InputStream response,
                         final TransportControl control) {
        Runnable task = new Runnable() {
            @Override
            public void run() {
                try {
                    cb.received(response, control);
                } catch (RuntimeException ex) {
                    cb.failed(ex);
                }
            }
        };
        executor.execute(task);
    }

    @Override
    public void failed(final RuntimeException error) {
        Runnable task = new Runnable() {
            @Override
            public void run() {
                cb.failed(error);
            }
        };
        executor.execute(task);
    }

    @Override
    public void completed() {
        Runnable task = new Runnable() {
            @Override
            public void run() {
                cb.completed();
            }
        };
        executor.execute(task);
    }
}

