/* **********************************************************
 * Copyright (c) 2015-2016, 2019-2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.client.rpc.http;

import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import com.vmware.vapi.internal.util.StringUtils;
import com.vmware.vapi.internal.util.Validate;

/**
 * Various utilities.
 */
public class Utils {

    /**
     * Adds <code>value</code> to the list in the <code>theMap</code> at position <code>key</code>.
     * If the list doesn't yet exist it is created.
     *
     * @param key of the list to which <code>value</code> is added. May be <code>null</code>.
     * @param value that is added to the list at position <code>key</code>. May be <code>null</code>
     *        .
     * @param theMap must not be <code>null</code>.
     */
    public static <K, V> void addListEntryToMapOfLists(K key,
                                                       V value,
                                                       Map<K, List<V>> theMap) {
        Validate.notNull(theMap);
        List<V> values = theMap.get(key);
        if (values == null) {
            values = new ArrayList<>();
            theMap.put(key, values);
        }
        values.add(value);
    }

    /**
     * Appends <code>path</code> and <code>query</code> as if they were URI path and query strings.
     * That is the result is <code>path + '?' + query</code>. In the case where path already
     * contains the '?' character and then the result is <code>path + '&' + query</code>. In case
     * <code>query</code> is <code>null</code> or empty string the result is <code>path</code>.In
     * the case where path ends with '?' or '&' the result is <code>path + query</code>.
     *
     * @param path must not be <code>null</code>.
     * @param query may be <code>null</code> in which case the path is returned as is.
     * @return appended <code>path</code> and <code>query</code> separated by the appropriate
     *         character - either '?' or '&'.
     */
    public static String appendQueryToPath(String path,
                                           String query) {
        Validate.notNull(path);
        if (StringUtils.isNotBlank(query)) {
            StringBuilder builder = new StringBuilder(path.length() + 1 + query.length());
            builder.append(path);
            if (!(path.endsWith("&") || path.endsWith("?"))) {
                builder.append(path.contains("?") ? "&" : "?");
            }
            return builder.append(query).toString();
        } else {
            return path;
        }
    }

    /**
     * Splits a string on the first occurrence of a character.
     *
     * @param str to be split. Must not be <code>null</code>.
     * @param c character at the first occurrence of which to split.
     * @return an array of strings with length of 2. The first element of the array is the part of
     *         the string that is before the first occurrence of the character; the second element
     *         is the part after the first occurrence. If the character doesn't occur in the string
     *         then the first element is the whole string and the second is <code>null</code>.
     */
    public static String[] splitOnFirstOccurrenceOfChar(String str,
                                                        char c) {
        Validate.notNull(str);
        String[] result = new String[2];
        int firstIndexOfEquals = str.indexOf(c);
        if (firstIndexOfEquals > 0) {
            result[0] = str.substring(0, firstIndexOfEquals);
            result[1] = str.substring(firstIndexOfEquals + 1);
        } else {
            result[0] = str;
            result[1] = null;
        }
        return result;
    }

    /**
     * TODO - remove it once switching to java 8
     *
     * Concatenates Strings within a Collection delimited by a specified
     * delimiter.
     *
     * @param collection Collection<String>. Must not be null
     * @param delimiter string. Must not be null
     * @return String -the concatenated values with delimiters
     */
    public static String join(Collection<String> collection, String delimiter) {
        Validate.notNull(collection);
        Validate.notNull(delimiter);
        StringBuilder builder = new StringBuilder();
        for (String s : collection) {
            builder.append(s);
            builder.append(delimiter);
        }
        if (builder.length() > 0) {
            builder.delete(builder.length() - delimiter.length(),
                           builder.length());
        }
        return builder.toString();
    }
}
