/* **********************************************************
 * Copyright (c) 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.client.rpc.http.handle;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.util.concurrent.locks.ReentrantLock;

import org.apache.http.HttpResponse;
import org.apache.http.nio.IOControl;
import org.apache.http.nio.client.methods.AsyncByteConsumer;
import org.apache.http.protocol.HttpContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.diagnostics.LogDiagnosticUtil;
import com.vmware.vapi.diagnostics.LogDiagnosticsConfigurator;
import com.vmware.vapi.diagnostics.Slf4jMDCLogConfigurator;
import com.vmware.vapi.internal.core.abort.AbortHandle;
import com.vmware.vapi.internal.core.abort.AbortListener;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.ResponseCallback;
import com.vmware.vapi.internal.protocol.client.rpc.CorrelatingClient.TransportControl;
import com.vmware.vapi.internal.protocol.common.http.ApacheHttpClientExceptionTranslator;
import com.vmware.vapi.internal.protocol.common.http.BinaryInput;
import com.vmware.vapi.internal.protocol.common.http.FrameDeserializer;
import com.vmware.vapi.internal.protocol.common.http.impl.ByteBufferBinaryInput;
import com.vmware.vapi.internal.util.Validate;

/**
 * Consumer which treats the response HTTP entity as a stream of frames and acts
 * as an AbortListener delegating the streaming cancelation to the
 * TransportControl.
 */
public class NioStreamingResponseConsumer
        extends AsyncByteConsumer<HttpResponse> implements AbortListener {

    private volatile HttpResponse httpResponse;
    private final ResponseCallback callback;
    private final FrameDeserializer deserializer;
    private final AbortHandle abortHandle;
    private final ExecutionContext ctx;

    private TransportControl transportControl;

    private static Logger logger = LoggerFactory
            .getLogger(NioStreamingResponseConsumer.class);

    private enum AbortState {
        INITIAL,
        REGISTERED,
        ABORTED;
    }

    private volatile AbortState abortState;

    /**
     * Multiple frames handled by a reactive streaming contract.
     *
     * @param deserializer - not null
     * @param callback - not null
     * @param abortHandle
     */
    public NioStreamingResponseConsumer(FrameDeserializer deserializer,
                                        ResponseCallback callback,
                                        AbortHandle abortHandle,
                                        ExecutionContext ctx) {
        Validate.notNull(deserializer);
        Validate.notNull(callback);
        this.deserializer = deserializer;
        this.callback = callback;
        this.ctx = ctx;
        this.abortHandle = abortHandle;
        this.abortState = AbortState.INITIAL;
    }

    @Override
    protected void onResponseReceived(final HttpResponse response) {
        httpResponse = response;
    }

    @Override
    protected void onByteReceived(final ByteBuffer buf, final IOControl ioctrl)
            throws IOException {
        LogDiagnosticsConfigurator logConfig = new Slf4jMDCLogConfigurator();
        try {
            logConfig.configureContext(LogDiagnosticUtil
                    .getDiagnosticContext(ctx));
            logger.debug("OnByteReceived event triggered.");

            ioctrl.suspendInput();

            ByteBufferBinaryInput input = new ByteBufferBinaryInput(buf
                    .asReadOnlyBuffer());
            TransportControl control = new NioControl(ioctrl,
                                                      input,
                                                      callback,
                                                      abortHandle,
                                                      deserializer,
                                                      ctx);
            registerStreamingAbortListener(control);

            control.resumeRead();
        } finally {
            logConfig.cleanUpContext(LogDiagnosticUtil.getDiagnosticKeys());
        }
    }

    private void registerStreamingAbortListener(TransportControl control) {
        this.transportControl = control;
        logger.trace("Updated transport control reference - %h.");

        if (abortHandle == null) {
            logger.debug("Abort listener not registered - no abort handle is supplied.");
            return;
        }

        if (abortState.equals(AbortState.INITIAL)) {
            logger.trace("AbortListener is being attached.");
            abortHandle.addAbortListener(this);
            abortState = AbortState.REGISTERED;
            if (abortHandle.isAborted()) {
                onAbort();
            }
        }
    }

    @Override
    protected HttpResponse buildResult(final HttpContext context) {
        LogDiagnosticsConfigurator logConfig = new Slf4jMDCLogConfigurator();
        try {
            logConfig.configureContext(LogDiagnosticUtil
                    .getDiagnosticContext(ctx));
            logger.debug("Streaming HTTP response complete");
        } finally {
            logConfig.cleanUpContext(LogDiagnosticUtil.getDiagnosticKeys());
        }
        return httpResponse;
    }

    @Override
    public synchronized void onAbort() {
        logger.trace("On abort event has been invoked.");
        if (abortState.equals(AbortState.REGISTERED)) {
            try {
                logger.debug("Shutting down streaming channel.");
                transportControl.cancel();
            } catch (Exception e) {
                logger.debug("IOControl shutdown failed.", e);
            } finally {
                abortState = AbortState.ABORTED;
            }
        }
    }

    /**
     * Exposes Apache NIO I/O controls.
     */
    private static class NioControl
            implements CorrelatingClient.TransportControl {
        private final FrameDeserializer deserializer;
        private final ExecutionContext ctx;

        private IOControl ioControl;
        private ByteBufferBinaryInput input;
        private ResponseCallback cb;
        private AbortHandle abortHandle;

        private volatile boolean aborted = false;
        private volatile boolean finished = false;

        private volatile boolean readSuspended = false;

        // Used for synchronizing different dispatcher threads
        private ReentrantLock lock;

        public NioControl(IOControl ioControl,
                          ByteBufferBinaryInput input,
                          ResponseCallback cb,
                          AbortHandle abortHandle,
                          FrameDeserializer deselrializer,
                          ExecutionContext ctx) {
            this.ioControl = ioControl;
            this.input = input;
            this.cb = cb;
            this.deserializer = deselrializer;
            this.ctx = ctx;
            this.abortHandle = abortHandle;

            this.lock = new ReentrantLock();
        }

        @Override
        public void suspendRead() {
            logger.trace(String
                    .format("Suspending read in TransportControl - %h.", this));
            readSuspended = true;
        }

        @Override
        public void resumeRead() {
            logger.trace(String
                    .format("Resuming read in TransportControl - %h.", this));
            LogDiagnosticsConfigurator logConfig = new Slf4jMDCLogConfigurator();

            if (lock.isHeldByCurrentThread()) {
                readSuspended = false;
                logger.trace(String
                        .format("The same thread is already reading frames in TransportControl - %h.",
                                this));
                return;
            }

            lock.lock();
            readSuspended = false;
            byte[] frame = null;
            try {
                if (input != null) {
                    logConfig.configureContext(LogDiagnosticUtil
                            .getDiagnosticContext(ctx));
                    while (!readSuspended && !aborted) {
                        logger.trace(String
                                .format("Reading frames in TransportControl - %h.",
                                        this));
                        frame = readFrame(input);

                        if ((frame == null || frame.length == 0)) {
                            input = null;
                            logger.debug(String
                                    .format("Finished reading - request additional input in TransportControl - %h.",
                                            this));
                            finished = true;
                            ioControl.requestInput();
                            break;
                        }
                        cb.received(new ByteArrayInputStream(frame), this);
                    }
                }
            } catch (IOException ex) {
                logger.debug("Error during reading frames.", ex);
                cb.failed(ApacheHttpClientExceptionTranslator
                        .translate(ex, abortHandle, ""));
            } finally {
                lock.unlock();
                logConfig.cleanUpContext(LogDiagnosticUtil.getDiagnosticKeys());
            }
        }

        private byte[] readFrame(BinaryInput input) throws IOException {
            synchronized (deserializer) {
                return deserializer.readFrame(input);
            }
        }

        @Override
        public void cancel() {
            logger.debug(String
                    .format("Cancelling the streaming frame desirialization in TransportControl - %h.",
                            this));
            try {
                // If the streaming is finished, then we do not want to shutdown
                // the connection, because it might be reused
                if (!finished) {
                    logger.debug(String
                            .format("Shutting down underlying channel in TransportControl - %h.",
                                    this));
                    ioControl.shutdown();
                }
                aborted = true;
                if (lock.tryLock()) {
                    input = null;
                    lock.unlock();
                }
                logger.debug(String
                        .format("Streaming frame desirialization cancelled in TransportControl - %h.",
                                this));
            } catch (IOException e) {
                logger.warn("Exception while trying to shutdown the IOcontrol",
                            e);
            }
        }
    }
}