/* **********************************************************
 * Copyright (c) 2016, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.common.http;

import java.util.Iterator;
import java.util.Map;
import java.util.Map.Entry;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import org.apache.commons.codec.EncoderException;
import org.apache.commons.codec.net.URLCodec;

import com.vmware.vapi.internal.util.StringUtils;
import com.vmware.vapi.internal.util.Validate;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 *
 */
public class UrlUtil {
    private static final Logger LOGGER = LoggerFactory.getLogger(UrlUtil.class);
    private static final String URI_PATH_SEPARATOR = "/";
    private static final Pattern pattern = Pattern.compile("\\{(.+?)\\}");
    /* TODO: rest native: get the QueryParamUrlCodec here?
     *       1) it is about RFC 3986 (percent encoding in URI)
     *       https://tools.ietf.org/html/rfc7230
     *       2) vs application/x-www-form-urlencoded
     *       https://www.w3.org/TR/html5/forms.html#url-encoded-form-data
     *
     *       - use the Java build-in URLEncoder?
     *       - don't %-encode chars which doesn't need to be
     *       - don't add + for space (but %20 instead)
     */
    private final URLCodec urlCodec = new URLCodec();

    /**
     * Replaces the path variables into the provided URL template. If there is no
     * value specified for a particular path variable placeholder then that
     * placaholder will not be replaced.
     *
     * <pre>
     * Example:
     *
     * template=/com/vmware/{id}, {id=vm-10}, result=/com/vmware/vm-10
     * template=/com/vmware/{type}/{id}, {id=vm-10}, result=/com/vmware/{type}/vm-10
     * template=/com/vmware/id, {id=vm-10}, result=/com/vmware/id
     * template=/com/vmware/{type}/{id}, {id=vm-10,type=vm}, result=/com/vmware/vm/vm-10
     * </pre>
     *
     * @param template URL template with placeholders to be replaced
     * @param vars path variables values
     * @return result URL after replacing the path variables placeholders
     */
    public String replacePathVariables(String template, Map<String, String> vars) {
        LOGGER.trace("Replace path variables called: template={}, vars={}", template, vars);
        Matcher matcher = pattern.matcher(template);
        StringBuilder builder = new StringBuilder();

        int parseIx = 0;
        while (matcher.find()) {
            String key = matcher.group().substring(1, matcher.group().length() - 1);
            String val = vars.get(key);
            LOGGER.trace("Found template key: {}", key);
            if (val == null) {
                LOGGER.trace("No value specified for key '{}', skipping replace.", key);
                continue;
            }

            LOGGER.trace("Replacing key '{}' with value '{}'", key, val);

            builder.append(template.substring(parseIx, matcher.start()));
            builder.append(val);
            parseIx = matcher.end();
        }

        if (parseIx < template.length() - 1) {
            builder.append(template.substring(parseIx));
        }

        return builder.toString();
    }

    /**
     * Creates path using provided base URL + one or more paths.
     * <p>
     * Use to join path as a alternative to simple concatenation. If base URL is
     * not set then only joined paths are returned.
     *
     * @param baseUrl The base URL, usually starting
     * @param paths to be joined
     * @return URL built using <code>baseUrl</code> plus joined paths.
     */
    public String joinUrls(String baseUrl, String... paths) {
        if (baseUrl == null) {
            baseUrl = StringUtils.EMPTY;
        }

        StringBuilder result = new StringBuilder();

        result.append(StringUtils.strip(baseUrl, URI_PATH_SEPARATOR));

        for (String path : paths) {
            result.append(URI_PATH_SEPARATOR);
            result.append(StringUtils.strip(path, URI_PATH_SEPARATOR));
        }

        return result.toString();
    }

    /**
     * Encodes a string into its URL safe form using the default string charset.
     * Unsafe characters are escaped. Note that this method escapes forward
     * slashes too.
     *
     * @param path
     *            string to convert to a URL safe form
     * @return URL safe string
     */
    public String encodeUrlPath(String path) {
        try {
            return urlCodec.encode(path);
        } catch (EncoderException e) {
            throw new IllegalArgumentException(e);
        }
    }

    /**
     * Encodes parameters to query string. Assumes that the parameters are already serialized to
     * String form but not URL encoded
     *
     * @param params values to encode
     * @return encoded HTTP request query string. May be empty string but never <code>null</code>.
     * @throws IllegalArgumentException if one of the keys in the map is <code>null</code> OR in
     *         case the key or value cannot be URL encoded
     */
    public String encodeQuery(Map<String, String> params) {
        // Consider estimating the size of the StringBuilder to optimize
        // performance
        StringBuilder resultBuilder = new StringBuilder();
        Iterator<Map.Entry<String, String>> iter = params.entrySet().iterator();
        while (iter.hasNext()) {
            Entry<String, String> param = iter.next();
            String key = param.getKey();
            String value = param.getValue();
            Validate.notNull(key);
            resultBuilder.append(encodeUrlPath(key));
            if (value != null) {
                resultBuilder.append("=").append(encodeUrlPath(value));
            }
            if (iter.hasNext()) {
                resultBuilder.append("&");
            }
        }
        return resultBuilder.toString();
    }
}
