/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common.http.impl;

import java.io.EOFException;
import java.io.IOException;

import com.vmware.vapi.internal.protocol.common.http.BinaryInput;
import com.vmware.vapi.internal.util.Validate;

/**
 * A deserializer which reads a single fixed-length data field.
 * <p>
 * <i>Thread-safety:</i> not thread-safe.
 * </p>
 */
public final class FixedLengthFieldDeserializer {

    private final int capacity;
    private final byte[] data;
    private int currentSize;
    private boolean done;

    public FixedLengthFieldDeserializer(int length) {
        Validate.isTrue(length >= 0);
        this.capacity = length;
        this.data = new byte[capacity];
        this.currentSize = 0;
        this.done = false;
    }

    /**
     * Reads a fixed-length field from an input handle. Should not call this
     * method after it returns a non-null value, because the deserializer can be
     * used only for a single field.
     *
     * @param input handle to an input source
     * @return the whole field or <code>null</code> if there is not enough data
     *         to construct a whole field
     * @throws IOException I/O error
     * @throws EOFException if the input source is exhausted permanently before
     *             reading the whole field
     * @throws IllegalStateException if an attempt is made to use the
     *             deserializer to read multiple fields
     */
    public byte[] read(BinaryInput input) throws IOException {
        if (done) {
            throw new IllegalStateException(
                    "Field already extracted. The deserializer can be used only for a single field.");
        }
        for (;;) {
            int off = currentSize;
            int remaining = capacity - currentSize;
            int readCount = input.read(data, off, remaining);

            if (readCount == BinaryInput.NO_DATA_TEMPORARY) {
                return null;
            }
            if (readCount == BinaryInput.NO_DATA_PERMANENT) {
                throw new EOFException();
            }
            if (readCount == remaining) {
                done = true;
                return data;
            }

            currentSize += readCount;
        }
    }

}
