/* **********************************************************
 * Copyright (c) 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common.json;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.vmware.vapi.data.BlobValue;
import com.vmware.vapi.data.BooleanValue;
import com.vmware.vapi.data.DataType;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DoubleValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.data.IntegerValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.SecretValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.data.VoidValue;

public class DefaultDataValueDeserializer implements JsonDataValueDeserializer {


    @Override
    public DataValue deserializeDataValue(JsonParser jp) throws IOException {
        return deserializeDV(jp);
    }

    @Override
    public ErrorValue deserializeErrorValue(JsonParser jp) throws IOException {
        nextExpectedToken(jp, JsonToken.FIELD_NAME);
        return deserializeEV(jp);
    }


    /**
     * Parses JSON representation for arbitrary <code>DataValue</code>. Which
     * is one of JSON Array (ListValue), JSON Object (OptionalValue, SecretValue,
     * BinaryValue, StructValue, ErrorValue) or JSON primitive (VoidValue,
     * IntegerValue, DoubleValue, StringValue). See the other deserializeXXX
     * methods for more details.
     *
     * The JSON stream <code>jp</code> must be positioned on first/proper JSON
     * token for the value being parsed, as defined by other deserializeXXX
     * methods. Upon this method completion the stream is positioned on
     * the token that follows the last token for the value being parsed.
     *
     * @param jp
     * @return
     * @throws JsonInvalidDataValueException if invalid JSON sequence/content
     *         is detected
     * @throws IOException
     */
    public static DataValue deserializeDV(JsonParser jp) throws IOException {
        if(jp.getCurrentToken() == JsonToken.START_ARRAY) {
            return deserializeListValue(jp);
        }

        if (jp.getCurrentToken() == JsonToken.START_OBJECT) {
            jp.nextToken();
        }

        if (jp.getCurrentToken() == JsonToken.FIELD_NAME) {
            String dataType = jp.getCurrentName();
            if (dataType.equals(DataType.STRUCTURE.getValue())) {
                return deserializeStructValue(jp);
            } else if (dataType.equals(DataType.OPTIONAL.getValue())) {
                return deserializeOptionalValue(jp);
            } else if (dataType.equals(DataType.SECRET.getValue())) {
                return deserializeSecretValue(jp);
            } else if (dataType.equals(DataType.BLOB.getValue())) {
                return deserializeBinaryValue(jp);
            } if (dataType.equals(DataType.ERROR.getValue())) {
                return deserializeEV(jp);
            } else {
                throw new JsonInvalidDataValueException(
                        "Unexpected JSON field detected: " + dataType);
            }
        }

        return deserializePrimitiveValue(jp);
    }

    /**
     * Parses JSON for <code>StructValue</code>.
     *
     * <pre>
     *   {
     *      "STRUCTURE": {
     *         "sample.first.count.properties": {
     *            "inc": 10,
     *            "dec": 10
     *         }
     *      }
     *   }
     *
     * </pre>
     *
     * The JSON stream <code>jp</code> must be positioned on the "STRUCTURE"
     * JSON field. Upon this method completion the stream is positioned on
     * the token that follows the last closing }.
     */
    static StructValue deserializeStructValue(JsonParser jp) throws JsonParseException, IOException {
        //Go Past the STRUCTURE:{ and position on the fieldName
        nextExpectedToken(jp, JsonToken.START_OBJECT);
        nextExpectedToken(jp, JsonToken.FIELD_NAME);
        StructValue sv = new StructValue(jp.getCurrentName());
        //Go Past the structure name and position on the fields of the structure
        nextExpectedToken(jp, JsonToken.START_OBJECT);
        jp.nextToken();
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            String fieldName = jp.getCurrentName();
            sv.setField(fieldName, deserializeDV(jp));
        }
        nextExpectedToken(jp, JsonToken.END_OBJECT);  // now current is closing } for "STRUCTURE": {
        jp.nextToken();                               // position on following one
        return sv;
    }

    /**
     * Parses JSON for <code>ListValue</code>.
     *
     * <pre>
     *   [ (any DataValue JSON), (any DataValue JSON), (...) ]
     *
     *   e.g.
     *   [ 1, 2, 3, 4 ]
     *   [ {"OPTIONAL": true}, {"OPTIONAL": false} ]
     * </pre>
     *
     * The JSON stream <code>jp</code> must be positioned on the opening [.
     * Upon this method completion the stream is positioned on the token that
     * follows the closing ].
     */
    static ListValue deserializeListValue(JsonParser jp)
            throws JsonParseException, IOException {
        ListValue lv = new ListValue();
        jp.nextToken();
        while (jp.getCurrentToken() != JsonToken.END_ARRAY) {
            if (jp.getCurrentToken() == JsonToken.START_OBJECT ||
                    jp.getCurrentToken() == JsonToken.START_ARRAY) {
                lv.add(deserializeDV(jp));
            } else {
                lv.add(deserializePrimitiveValue(jp));
            }
        }
        jp.nextToken();  // position on token after the list closing ]
        return lv;
    }

    /**
     * Parses JSON for primitive value:
     * <ul>
     * <li><code>BooleanValue</code>
     * <li><code>IntegerValue</code>
     * <li><code>DoubleValue</code>
     * <li><code>StringValue</code>
     * <li><code>VoidValue</code>
     *
     * <pre>
     *   true
     *   42
     *   7.43
     *   "some string value"
     *   null
     * </pre>
     *
     * The JSON stream <code>jp</code> must be positioned on the JSON token
     * representing the value. Upon this method completion the stream is
     * position on the following token.
     */
    static DataValue deserializePrimitiveValue(JsonParser jp) throws IOException {
        try {
            DataValue dv = null;
            if (jp.getBooleanValue()) {
                dv = BooleanValue.TRUE;
            } else {
                dv = BooleanValue.FALSE;
            }
            jp.nextToken();
            return dv;
        } catch (JsonParseException jpEx) {
            if (jp.getCurrentToken() == JsonToken.FIELD_NAME) {
                jp.nextValue();
            }
            JsonToken curr = jp.getCurrentToken();
            DataValue dv = null;
            if (curr == JsonToken.VALUE_NUMBER_INT) {
                dv = new IntegerValue(jp.getLongValue());
            } else if (curr == JsonToken.VALUE_NUMBER_FLOAT) {
                // TODO: if it is not in the canonical double format for JSON
                //       report an error!
                dv = new DoubleValue(jp.getDoubleValue());
            } else if (curr == JsonToken.VALUE_NULL) {
                dv = VoidValue.getInstance();
            } else {
                dv = new StringValue(jp.getText());
            }
            jp.nextToken(); // position on the following token
            return dv;
        }
    }

    /**
     * Parses JSON for <code>OptionalValue</code>.
     *
     * <pre>
     *   { "OPTIONAL": (any DataValue JSON) }
     *
     *   e.g.
     *   { "OPTIONAL" : "str1" }
     *   { "OPTIONAL" : null }
     * </pre>
     *
     * The JSON stream <code>jp</code> must be positioned on the "OPTIONAL"
     * JSON field. Upon this method completion the stream is positioned on
     * the token that follows the closing }.
     */
    static OptionalValue deserializeOptionalValue(JsonParser jp)
             throws JsonParseException, IOException {
        if (jp.nextToken() == JsonToken.VALUE_NULL) {
            nextExpectedToken(jp, JsonToken.END_OBJECT);
            jp.nextToken();   // position on the following token
            return OptionalValue.UNSET;
        } else {
            OptionalValue result = new OptionalValue(deserializeDV(jp));
            jp.nextToken();  // position on the following token
            return result;
        }
    }

    /**
     * Parses JSON for <code>SecretValue</code>.
     *
     * <pre>
     *   { "SECRET": "some_secret" }
     * </pre>
     *
     * The JSON stream <code>jp</code> must be positioned on the opening {.
     * Upon this method completion the stream is positioned on the token that
     * follows the closing }.
     */
    static SecretValue deserializeSecretValue(JsonParser jp)
            throws JsonParseException, IOException {
        jp.nextToken();
        String text = jp.getText();
        nextExpectedToken(jp, JsonToken.END_OBJECT);
        jp.nextToken();  // position on the following token
        return new SecretValue(text.toCharArray());
    }

    /**
     * Parses JSON for <code>BlobValue</code>.
     *
     * <pre>
     *   { "BINARY": "AQ0PAP0q" }
     * </pre>
     *
     * The JSON stream <code>jp</code> must be positioned on the opening {.
     * Upon this method completion the stream is positioned on the token that
     * follows the closing }.
     */
    static BlobValue deserializeBinaryValue(JsonParser jp)
            throws JsonParseException, IOException {
        jp.nextToken();
        byte[] bytes = jp.getBinaryValue();
        nextExpectedToken(jp, JsonToken.END_OBJECT);
        jp.nextToken();  // position on the following token
        return new BlobValue(bytes);
    }

    /**
     * Parses JSON for <code>ErrorValue</code>.
     *
     * <pre>
     *   {
     *      "ERROR": {
     *         "com.vmware.vapi.std.not_found": {
     *            "messages": ( Any DataValue JSON )
     *         }
     *      }
     *   }
     *
     * </pre>
     *
     * The JSON stream <code>jp</code> must be positioned on the "ERROR"
     * JSON field. Upon this method completion the stream is positioned on
     * the token that follows the last closing }.
     */
    static ErrorValue deserializeEV(JsonParser jp)
            throws JsonParseException, IOException {
        // TODO: this code is duplicated in deserializeStructValue
        nextExpectedToken(jp, JsonToken.START_OBJECT);
        nextExpectedToken(jp, JsonToken.FIELD_NAME);
        ErrorValue val = new ErrorValue(jp.getText());
        nextExpectedToken(jp, JsonToken.START_OBJECT);
        jp.nextToken();
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            String fieldName = jp.getCurrentName();
            val.setField(fieldName, deserializeDV(jp));
        }
        nextExpectedToken(jp, JsonToken.END_OBJECT);  // now current is closing } for "ERROR": {
        jp.nextToken();                               // position on following one
        return val;
    }


    /**
     * Advances the <code>jp</code> to the next token and verifies that it
     * is of type <code>expected</code>.
     *
     * @param jp
     * @param expected
     * @throws IOException
     * @throws JsonParseException
     * @throws JsonInvalidDataValueException if the next token is not of the
     *         expected type
     */
    static void nextExpectedToken(JsonParser jp, JsonToken expected)
            throws JsonParseException, IOException {

        if (jp.nextToken() != expected) {
            throw new JsonInvalidDataValueException(
                    String.format("Expected %s JSON token but detected %s",
                                  expected, jp.getCurrentToken()));
        }
    }


}
