/* **********************************************************
 * Copyright (c) 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.protocol.common.json;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorValue;

/**
 * Interface for JSON based {@link DataValue} de-serialization logic.
 * <p>
 * This interface only reads a DataValue and cannot process message envelope. IT
 * should be used in conjunction with a class that initializes the
 * {@link JsonParser} and processes any envelope data surrounding the
 * {@link DataValue}
 */
public interface JsonDataValueDeserializer {
    /**
     * Reads {@link DataValue} from a {@link JsonParser}
     * @param jp JSON parser position at the start of a {@link DataValue}
     * @return de-serialized {@link DataValue} object
     * @throws IOException if consuming the underlying data fails
     * @throws JsonParseException if reading JSON fails
     * @throws JsonInvalidDataValueException if invalid JSON sequence/content
     *         is detected
     */
    DataValue deserializeDataValue(JsonParser jp) throws IOException;

    /**
     * Reads {@link ErrorValue} from a {@link JsonParser}
     * @param jp JSON parser position at the start of a {@link ErrorValue}
     * @return de-serialized {@link DataValue} object
     * @throws IOException if consuming the underlying data fails
     * @throws JsonParseException if reading JSON fails
     * @throws JsonInvalidDataValueException if invalid JSON sequence/content
     *         is detected
     */
    ErrorValue deserializeErrorValue(JsonParser jp) throws IOException;
}
