/* **********************************************************
 * Copyright (c) 2013-2017, 2019-2020 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.common.json;

import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.APP_CTX;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.CONTEXT;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_ID;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_JSONRPC;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_METHOD;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_OPERATION_ID;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_PARAMS;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_RESULT;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_SERVICE_ID;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.INPUT;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.JSON_ERROR_CODE;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.JSON_ERROR_DATA;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.JSON_ERROR_MESSAGE;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.SEC_CTX;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.JsonToken;
import com.vmware.vapi.CoreException;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.core.ExecutionContext.SecurityContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.internal.protocol.common.json.JsonConstants.RequestType;
import com.vmware.vapi.internal.protocol.common.json.JsonContextBuilderRequestParams.ExecutionContextBuilder;
import com.vmware.vapi.protocol.common.json.JsonRpcDeserializer;

/**
 * JSON-RPC Deserializer for vAPI wire protocol.
 */
public class JsonMsgDeserializer2
        implements JsonDeserializer, JsonRpcDeserializer {
    private static final String OUTPUT = "output";
    private static final String LAST = "last";
    private static final String ERROR = "error";
    private static final String PROGRESS = "progress";

    private static final String RPC_VERSION = "2.0";

    private static final String INVALID_TYPE_IN_SECURITY_CONTEXT_MSG = "Found invalid type in the security context";

    private final JsonFactory factory;

    private final JsonDataValueDeserializer dvDeserializer;

    /**
     * Initializes Jackson JsonFactory that will be used to create JsonParser
     * instances to deserialize requests/responses.
     * <p>
     * This constructor initializes the default VAPI JSON deserializer with
     * embedded type information.
     */
    public JsonMsgDeserializer2() {
        this(new DefaultDataValueDeserializer());
    }

    /**
     * Initializes Jackson JsonFactory that will be used to create JsonParser
     * instances to deserialize requests/responses.
     *
     * @param dvDeserializer desired {@link JsonDataValueDeserializer}. VAPI
     *        ships with {@link DefaultDataValueDeserializer} and
     *        {@link JsonDirectDeserializer}
     */
    public JsonMsgDeserializer2(JsonDataValueDeserializer dvDeserializer) {
        this.factory = new JsonFactory();
        this.dvDeserializer = dvDeserializer;
    }

    /**
     * Method to deserialize a JSON-RPC 2.0 request. The JSON-RPC method
     * specified must be "invoke".
     *
     * @param jsonRequest Incoming JSON-RPC 2.0 request
     * @return an instance of JsonApiRequest
     * @throws JsonInvalidMethodParamsException
     * @throws JsonInvalidRequest
     * @throws JsonInvalidMethodException
     * @throws JsonParseException
     * @throws IOException
     */
    @Override
    public JsonApiRequest requestDeserialize(String jsonRequest)
            throws JsonInvalidMethodParamsException, JsonInvalidRequest,
            JsonInvalidDataValueException, JsonInvalidMethodException,
            JsonParseException, IOException {
        try (JsonParser jp = factory.createParser(jsonRequest)) {
            String rpcVersion = null;
            String id = null;
            String jsonMethod = null;
            JsonInvokeParams params = null;
            jp.nextToken();
            JsonToken nextToken;
            while ((nextToken = jp.nextToken()) != JsonToken.END_OBJECT) {
                if (nextToken == null) {
                    // null means end of stream, just stop processing
                    break;
                }
                JsonToken token = jp.getCurrentToken();
                if (token != JsonToken.FIELD_NAME) {
                    throw new JsonInvalidRequest(String
                            .format("Received unexpected json token %s", token),
                                                 id);
                }
                String jsonNode = jp.getCurrentName();
                if (jsonNode.equals(FIELD_JSONRPC)) {
                    jp.nextToken();
                    // TODO: from old deser: This can also be an integer.
                    // Need to handle/test that case
                    rpcVersion = jp.getText();
                    if (!rpcVersion.equals(RPC_VERSION)) {
                        throw new JsonInvalidRequest(String
                                .format("Received Invalid rpc version %s",
                                        rpcVersion), id);
                    }
                } else if (jsonNode.equals(FIELD_ID)) {
                    // TODO: from old deser: ID can also be a an integer. So,
                    // need to handle that case.
                    jp.nextToken();
                    id = jp.getText();
                } else if (jsonNode.equals(FIELD_METHOD)) {
                    jp.nextToken();
                    jsonMethod = jp.getText();
                    if (!RequestType.invoke.toString().equals(jsonMethod)) {
                        throw new JsonInvalidMethodException(id, jsonMethod,
                                "Invalid method:" + jsonMethod);
                    }
                } else if (jp.getText().equals(FIELD_PARAMS)) {
                    nextExpectedToken(jp, JsonToken.START_OBJECT);
                    nextExpectedToken(jp, JsonToken.FIELD_NAME); // read next
                                                                 // field (of
                                                                 // "params" : {
                                                                 // )

                    ExecutionContextBuilder ctx = null;
                    String serviceId = null;
                    String operationId = null;
                    DataValue dv = null;
                    while ((token = jp
                            .getCurrentToken()) != JsonToken.END_OBJECT) {
                        if (token != JsonToken.FIELD_NAME) {
                            throw new JsonInvalidRequest(String
                                    .format("Received unexpected json token %s",
                                            token), id);
                        }
                        String fieldName = jp.getCurrentName();
                        if (CONTEXT.equals(fieldName)) {
                            jp.nextToken();
                            ctx = deserializeExCtx(jp, id);
                            jp.nextToken();
                        } else if (FIELD_SERVICE_ID.equals(fieldName)) {
                            jp.nextToken();
                            serviceId = jp.getText();
                            jp.nextToken();
                        } else if (FIELD_OPERATION_ID.equals(fieldName)) {
                            jp.nextToken();
                            operationId = jp.getText();
                            jp.nextToken();
                        } else if (INPUT.equals(fieldName)) {
                            jp.nextToken();
                            dv = dvDeserializer.deserializeDataValue(jp);
                        } else {
                            throw new JsonInvalidRequest(String
                                    .format("Unexpected field in '%s' value: %s",
                                            FIELD_PARAMS,
                                            fieldName), id);
                        }
                    }
                    // verify we have read all 4 json fields: serviceId,
                    // operationId, ctx and input
                    verifyFieldProcessed(FIELD_SERVICE_ID, serviceId, id);
                    verifyFieldProcessed(FIELD_OPERATION_ID, operationId, id);
                    verifyFieldProcessed(CONTEXT, ctx, id);
                    verifyFieldProcessed(INPUT, dv, id);

                    params = new JsonContextBuilderRequestParams(serviceId,
                                                                 operationId,
                                                                 ctx,
                                                                 dv);
                } else {
                    throw new JsonInvalidRequest(String
                            .format("Received unexpected jsonNode %s",
                                    jsonNode), id);
                }
            }
            // verify we have read all 4 json fields: jsonrpc, id, method,
            // params
            verifyFieldProcessed(FIELD_JSONRPC, rpcVersion, id);
            verifyFieldProcessed(FIELD_ID, id, id);
            verifyFieldProcessed(FIELD_METHOD, jsonMethod, id);
            verifyFieldProcessed(FIELD_PARAMS, params, id);

            return new JsonApiRequest(id, params);
        }
    }

    /**
     * Method to deserialize a JSON-RPC 2.0 response
     *
     * @param jsonResponse A JSON-RPC 2.0 response.
     * @param responseType Enum indicating the type of response
     * @return An instance of JsonBaseResponse or one of the extended classes
     */
    @Override
    public JsonBaseResponse responseDeserialize(InputStream jsonResponse,
                                                RequestType responseType) {
        try (JsonParser jp = factory.createParser(jsonResponse)) {
            String rpcVersion = null;
            String id = null;
            DataValue output = null;
            ErrorValue error = null;

            jp.nextToken();
            JsonToken nextToken;
            while ((nextToken = jp.nextToken()) != JsonToken.END_OBJECT) {
                if (nextToken == null) {
                    // null means end of stream, just stop processing
                    break;
                }
                JsonToken token = jp.getCurrentToken();
                if (token != JsonToken.FIELD_NAME) {
                    throw toVapiCoreException(new JsonInvalidResponse(
                            String.format("Received unexpected json token %s",
                                    token)));
                }
                String jsonNode = jp.getCurrentName();
                if (FIELD_JSONRPC.equals(jsonNode)) {
                    jp.nextToken();
                    rpcVersion = jp.getText();
                    if (!RPC_VERSION.equals(rpcVersion)) {
                        throw toVapiCoreException(new JsonInvalidResponse(
                                String.format("Received Invalid rpc version %s",
                                        rpcVersion)));
                    }
                } else if (FIELD_ID.equals(jsonNode)) {
                    jp.nextToken();
                    id = jp.getText();
                } else if (FIELD_RESULT.equals(jsonNode)) {
                    nextExpectedToken(jp, JsonToken.START_OBJECT);
                    nextToken = jp.nextToken();
                    if (nextToken == JsonToken.FIELD_NAME) {
                        String fieldName = jp.getCurrentName();
                        if (OUTPUT.equals(fieldName)) {
                            jp.nextToken();
                            output = dvDeserializer.deserializeDataValue(jp);
                            // top { "result" : { "error" :...} field - for vAPI
                            // error
                        } else if (ERROR.equals(fieldName)) {
                            nextExpectedToken(jp, JsonToken.START_OBJECT);
                            error = dvDeserializer.deserializeErrorValue(jp);
                        } else if (PROGRESS.equals(fieldName)) {
                            jp.nextToken();
                            return new JsonProgressResponse(id,
                                                            dvDeserializer
                                                                    .deserializeDataValue(jp));
                        } else {
                            throw new JsonInvalidDataValueException(String
                                    .format("Unexpected field %s", fieldName));
                        }
                    } else if (nextToken != JsonToken.END_OBJECT) {
                        throw new JsonInvalidDataValueException(String
                                .format("Expected empty object or response "
                                        + "field but detected %s",
                                        nextToken));
                    }
                    // top level "error" field - for JSON-RPC error
                } else if (ERROR.equals(jsonNode)) {
                    // TODO: extract method for this and others
                    String jsonErrorCode = null;
                    String jsonErrorMessage = null;
                    String jsonErrorData = null;
                    jp.nextToken();
                    while (jp.nextToken() != JsonToken.END_OBJECT) {
                        if (jp.getCurrentToken() != JsonToken.FIELD_NAME) {
                            throw toVapiCoreException(new JsonInvalidResponse(String
                                    .format("Received unexpected json token %s",
                                            jp.getCurrentToken())));
                        }

                        if (JSON_ERROR_CODE.equals(jp.getCurrentName())) {
                            jp.nextToken();
                            jsonErrorCode = jp.getText();
                        } else if (JSON_ERROR_MESSAGE
                                .equals(jp.getCurrentName())) {
                            jp.nextToken();
                            jsonErrorMessage = jp.getText();
                        } else if (JSON_ERROR_DATA
                                .equals(jp.getCurrentName())) {
                            jp.nextToken();
                            jsonErrorData = jp.getText();
                        }
                    }
                    // TODO: enable this
                    // make sure we read both "code" and "message" fields for
                    // JSON-RPC error
                    // verifyFieldProcessed(JSON_ERROR_CODE, jsonErrorCode, id);
                    // verifyFieldProcessed(JSON_ERROR_MESSAGE,
                    // jsonErrorMessage, id);
                    String errorMessage = "Recieved JSON-RPC 2.0 error with code:"
                                          + jsonErrorCode + " and message:"
                                          + jsonErrorMessage;
                    throw new JsonInvalidResponse(errorMessage,
                                                  jsonErrorCode,
                                                  jsonErrorData);

                } else {
                    throw toVapiCoreException(new JsonInvalidResponse(String
                            .format("Received unexpected jsonNode %s",
                                    jsonNode)));
                }
            }

            // verify we have read all 2 json fields: jsonrpc, id
            verifyFieldProcessed(FIELD_JSONRPC, rpcVersion, id);
            verifyFieldProcessed(FIELD_ID, id, id);
            // TODO: need to check we parsed either "result" or "error"
            // verifyFieldProcessed(FIELD_RESULT, <the_result>, id);

            if (output != null && error != null) {
                throw toVapiCoreException(new JsonInvalidResponse("Received both Result and Error values"));
            } else if (output == null && error == null) {
                return new JsonApiResponse(id, MethodResult.EMPTY);
                // TODO: If we get switch stream or no stream we can switch if
                // error is tossed on empty result
                // throw toVapiCoreException(new JsonInvalidResponse("Both
                // Result and Error Values are missing"));
            } else if (output != null) {
                return new JsonApiResponse(id, MethodResult.newResult(output));
            } else {
                return new JsonApiResponse(id,
                                           MethodResult.newErrorResult(error));
            }

        } catch (JsonInvalidResponse e) {
            throw toVapiCoreException(e);
        } catch (JsonParseException e) {
            throw toVapiCoreException(e);
        } catch (IOException e) {
            throw toVapiCoreException(e);
        }
    }

    @Override
    public DataValue deserializeDataValue(String jsonString) {
        try (JsonParser jp = factory.createParser(jsonString)) {
            jp.nextToken();
            return dvDeserializer.deserializeDataValue(jp);
        } catch (IOException ex) {
            throw toVapiCoreException(ex);
        }
    }

    private static ExecutionContextBuilder deserializeExCtx(JsonParser jp,
                                                            String id)
            throws JsonParseException, JsonInvalidMethodParamsException,
            IOException {

        ExecutionContextBuilder exCtx = new ExecutionContextBuilder();
        while (jp.nextToken() != JsonToken.END_OBJECT) {
            String fieldName = jp.getCurrentName();
            if (fieldName.equals(APP_CTX)) {
                jp.nextToken();
                while (jp.nextToken() != JsonToken.END_OBJECT) {
                    String key = jp.getText();
                    jp.nextToken();
                    String value = jp.getText();
                    exCtx.applicationData.put(key.toLowerCase(Locale.ENGLISH),
                                              value);
                }
            } else if (fieldName.equals(SEC_CTX)) {
                jp.nextToken();
                deserializeSecurityContext(jp, id, exCtx.security);
            }
        }
        return exCtx;
    }

    private static void deserializeSecurityContext(JsonParser jp,
                                                   String id,
                                                   Map<String, Object> secCtxData)
            throws JsonParseException, JsonInvalidMethodParamsException,
            IOException {

        JsonToken currentToken = jp.getCurrentToken();
        if (currentToken != JsonToken.START_OBJECT) {
            return;
        }

        jp.nextToken();
        while (currentToken != JsonToken.END_OBJECT) {
            String key = jp.getText();
            jp.nextToken();
            Object value = parseJsonObject(jp);
            secCtxData.put(key, value);
            currentToken = jp.getCurrentToken();
        }
        // TODO: not sure why this is needed here, given that there is
        // specialized parser for the SecurityContext - need to refactor
        Object schemeId = secCtxData.get("schemeId");
        if (schemeId == null || schemeId.getClass() != String.class) {
            throw new JsonInvalidMethodParamsException(id,
                                                       "Missing or invalid value type for 'schemeId'");
        }
        secCtxData.put(SecurityContext.AUTHENTICATION_SCHEME_ID, schemeId);
    }

    private static Object parseJsonObject(JsonParser jp)
            throws JsonParseException, IOException {
        // TODO need to handle arrays in the future when needed.
        Object value = null;
        JsonToken curr = jp.getCurrentToken();
        if (curr == JsonToken.VALUE_STRING) {
            value = jp.getText();
        } else if (curr == JsonToken.START_OBJECT) {
            jp.nextToken();
            Map<String, Object> data = new HashMap<>();
            while (curr != JsonToken.END_OBJECT) {
                String key = jp.getText();
                jp.nextToken();
                Object val = parseJsonObject(jp);
                data.put(key, val);
                curr = jp.getCurrentToken();
            }
            value = data;
        } else {
            throw toVapiCoreException(new RuntimeException(INVALID_TYPE_IN_SECURITY_CONTEXT_MSG));
        }
        jp.nextToken();
        return value;
    }

    /**
     * Advances the <code>jp</code> to the next token and verifies that it is of
     * type <code>expected</code>.
     *
     * @param jp
     * @param expected
     * @throws IOException
     * @throws JsonParseException
     * @throws JsonInvalidDataValueException if the next token is not of the
     *         expected type
     */
    static void nextExpectedToken(JsonParser jp, JsonToken expected)
            throws JsonParseException, IOException {

        if (jp.nextToken() != expected) {
            throw new JsonInvalidDataValueException(String
                    .format("Expected %s JSON token but detected %s",
                            expected,
                            jp.getCurrentToken()));
        }
    }

    private static void verifyFieldProcessed(String fieldName,
                                             Object value,
                                             String id)
            throws JsonInvalidMethodParamsException {
        if (value == null) {
            throw new JsonInvalidMethodParamsException(id,
                                                       "Missing node: "
                                                           + fieldName);
        }
    }

    private static CoreException toVapiCoreException(Exception ex) {
        return new CoreException(MessageFactory
                .getMessage("vapi.json.deserialize.ioerror", ex.getMessage()),
                                 ex);
    }
}
