/* **********************************************************
 * Copyright (c) 2013, 2016-2017, 2018 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.common.json;

import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.CONTEXT;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_ID;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_JSONRPC;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_METHOD;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_OPERATION_ID;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_PARAMS;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_RESULT;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.FIELD_SERVICE_ID;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.INPUT;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.JSON_ERROR_CODE;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.JSON_ERROR_DATA;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.JSON_ERROR_MESSAGE;
import static com.vmware.vapi.internal.protocol.common.json.JsonConstants.JSON_RPC_VERSION;

import java.io.IOException;
import java.util.Collections;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonGenerator;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.Version;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializerProvider;
import com.fasterxml.jackson.databind.module.SimpleModule;
import com.fasterxml.jackson.databind.ser.std.StdSerializer;
import com.vmware.vapi.CoreException;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.ExecutionContext.ApplicationData;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.BlobValue;
import com.vmware.vapi.data.BooleanValue;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DoubleValue;
import com.vmware.vapi.data.ErrorValue;
import com.vmware.vapi.data.IntegerValue;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.SecretValue;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.data.VoidValue;
import com.vmware.vapi.internal.dsig.json.CanonicalizationUtil;
import com.vmware.vapi.protocol.common.json.JsonRpcSerializer;

/**
 * JSON Serializer for vAPI runtime objects.
 *
 * <i>Thread-safety:</i> Instances of this class are thread-safe.
 */
public class JsonMsgSerializer2 implements JsonSerializer, JsonRpcSerializer {

    private static final Logger logger =
        LoggerFactory.getLogger(JsonMsgSerializer2.class);

    private static SimpleModule module;
    static {
        // TODO: have to switch off the default serialization (when no of the
        //       serializers below is capable to seriqlize the input object);
        //       the default serialization will kick in and serialize all
        //       public fields/accessors;
        //       however this might not be what we want, so we better don't
        //       count on it.
        module = new SimpleModule("vAPI", new Version(0, 1, 0, null, null, null));
        module.addSerializer(new VoidValueSerializer());
        module.addSerializer(new BooleanValueSerializer());
        module.addSerializer(new IntegerValueSerializer());
        module.addSerializer(new DoubleValueSerializer());
        module.addSerializer(new StringValueSerializer());
        module.addSerializer(new BlobValueSerializer());
        module.addSerializer(new OptionalValueSerializer());
        module.addSerializer(new ListValueSerializer());
        module.addSerializer(new StructValueSerializer());
        module.addSerializer(new SecretValueSerializer());
        module.addSerializer(new BooleanValueSerializer());
        module.addSerializer(new ErrorValueSerializer());
        module.addSerializer(new MethodResultSerializer());
        module.addSerializer(new AppContextSerializer());
        module.addSerializer(new ApiRequestSerializer());
        module.addSerializer(new ApiResponseSerializer());
        module.addSerializer(new JsonErrorSerializer());
    }

    private static final String METHOD_RESULT_OUTPUT = "output";
    private static final String METHOD_RESULT_ERROR ="error";

    private final ObjectMapper mapper;

    public JsonMsgSerializer2() {
        mapper = new ObjectMapper();
        mapper.registerModule(module);
    }

    /**
     * The <code> VoidValueSerializer</code> class serializes a
     * <code>{@link VoidValue}</code> into JSON
     */
    @SuppressWarnings("serial")
    private static class VoidValueSerializer extends StdSerializer<VoidValue> {
        /** Constructor. */
        public VoidValueSerializer() {
            super(VoidValue.class);
        }

        /**
         * Serializes the provided <code>{@link VoidValue}</code> into a
         * JSON null value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param value     void value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(VoidValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeNull();
        }
    }

    /**
     * The <code>BooleanValueSerializer</code> class serializes a
     * <code>{@link BooleanValue}</code> into JSON
     */
    @SuppressWarnings("serial")
    private static class BooleanValueSerializer extends StdSerializer<BooleanValue> {
        /** Constructor. */
        public BooleanValueSerializer() {
            super(BooleanValue.class);
        }

        /**
         * Serializes the provided <code>{@link BooleanValue}</code> into
         * a JSON true or false value and writes the result using the
         * provided <code>{@link JsonGenerator}</code>.
         *
         * @param value     boolean value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(BooleanValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeBoolean(value.getValue());
        }
    }

    /**
     * The <code> IntegerValueSerializer</code> class serializes a
     * <code>{@link IntegerValue}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class IntegerValueSerializer extends StdSerializer<IntegerValue> {
        /** Constructor. */
        public IntegerValueSerializer() {
            super(IntegerValue.class);
        }

        /**
         * Serializes the provided <code>{@link IntegerValue}</code> into
         * a JSON numeric value and writes the result using the
         * provided <code>{@link JsonGenerator}</code>.
         *
         * @param value     integer value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(IntegerValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeNumber(value.getValue());
        }
    }

    /**
     * The <code> DoubleValueSerializer</code> class serializes a
     * <code>{@link DoubleValue}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class DoubleValueSerializer extends StdSerializer<DoubleValue> {
        /** Constructor. */
        public DoubleValueSerializer() {
            super(DoubleValue.class);
        }

        /**
         * Serializes the provided <code>{@link DoubleValue}</code> into
         * a JSON numeric value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * <p>The serialized double value is always in the canonical format
         * for JSON double, as defined in
         * {@link CanonicalizationUtil#canonicalizeDouble}.
         *
         * @param value     double value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(DoubleValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {

            jgen.writeNumber(
                    CanonicalizationUtil.canonicalizeDouble(value.getValue()));
        }
    }

    /**
     * The <code> StringValueSerializer</code> class serializes a
     * <code>{@link StringValue}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class StringValueSerializer extends StdSerializer<StringValue> {
        /** Constructor. */
        public StringValueSerializer() {
            super(StringValue.class);
        }

        /**
         * Serializes the provided <code>{@link StringValue}</code> into a
         * JSON string value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param value     string value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(StringValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeString(value.getValue());
        }
    }

    /**
     * The <code>BlobValueSerializer</code> class serializes a
     * <code>{@link BlobValue}</code> into JSON
     */
    @SuppressWarnings("serial")
    private static class BlobValueSerializer extends StdSerializer<BlobValue> {
        /** Constructor. */
        public BlobValueSerializer() {
            super(BlobValue.class);
        }

        /**
         * Serializes the provided <code>{@link BlobValue}</code> into a
         * base64-encoded JSON string and writes the result using the
         * provided <code>{@link JsonGenerator}</code>.
         *
         * @param value     blob value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(BlobValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();
            provider.defaultSerializeField(value.getType().getValue(), value.getValue(), jgen);
            jgen.writeEndObject();
        }
    }

    /**
     * The <code> OptionalValueSerializer</code> class serializes a
     * <code>{@link OptionalValue}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class OptionalValueSerializer extends StdSerializer<OptionalValue> {
        /** Constructor. */
        public OptionalValueSerializer() {
            super(OptionalValue.class);
        }

        /**
         * Serializes the provided <code>{@link OptionalValue}</code> into
         * a JSON value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param value     optional value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(OptionalValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();
            jgen.writeFieldName(value.getType().getValue());
            if (value.isSet()) {
                jgen.writeObject(value.getValue());
            } else {
                jgen.writeNull();
            }
            // End of OptionalValue object
            jgen.writeEndObject();
        }
    }

    /**
     * The <code> ListValueSerializer</code> class serializes a
     * <code>{@link ListValue}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class ListValueSerializer extends StdSerializer<ListValue> {
        /** Constructor. */
        public ListValueSerializer() {
            super(ListValue.class);
        }

        /**
         * Serializes the provided <code>{@link ListValue}</code> into
         * a JSON array value and writes the result using the
         * provided <code>{@link JsonGenerator}</code>.
         *
         * @param value     list value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(ListValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartArray();
            for (DataValue serializeValue : value.getList()) {
                jgen.writeObject(serializeValue);
            }
            jgen.writeEndArray();
        }
    }

    /**
     * The <code> StructValueSerializer</code> class serializes a
     * <code>{@link StructValue}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class StructValueSerializer extends StdSerializer<StructValue> {
        /** Constructor. */
        public StructValueSerializer() {
            super(StructValue.class);
        }

        /**
         * Serializes the provided <code>{@link StructValue}</code> into a
         * JSON object value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param value     struct value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(StructValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();
            jgen.writeObjectFieldStart(value.getType().getValue());
            jgen.writeObjectField(value.getName(), value.getFields());
            // End of embedded StructValue
            jgen.writeEndObject();
            // End of StructValue object
            jgen.writeEndObject();
        }
    }

    /**
     * The <code> ErrorValueSerializer</code> class serializes a
     * <code>{@link ErrorValue}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class ErrorValueSerializer extends StdSerializer<ErrorValue> {
        /** Constructor. */
        public ErrorValueSerializer() {
            super(ErrorValue.class);
        }

        /**
         * Serializes the provided <code>{@link ErrorValue}</code> into a
         * JSON object value and writes the result using the provided
         * {@link JsonGenerator}.
         *
         * @param error     ErrorValue to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(ErrorValue error,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();
            jgen.writeObjectFieldStart(error.getType().getValue());
            jgen.writeObjectField(error.getName(), error.getFields());
            // End of embedded ErrorValue
            jgen.writeEndObject();
            // End of ErrorValue object
            jgen.writeEndObject();
        }
    }


    /**
     * The <code> SecretValueSerializer</code> class serializes a
     * <code>{@link SecretValue}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class SecretValueSerializer extends StdSerializer<SecretValue> {
        /** Constructor. */
        public SecretValueSerializer() {
            super(SecretValue.class);
        }

        /**
         * Serializes the provided <code>{@link SecretValue}</code> into a
         * JSON object value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param value     Secret value to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(SecretValue value,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();
            jgen.writeStringField(value.getType().getValue(), new String(value.getValue()));
            jgen.writeEndObject();
        }
    }

    /**
     * The <code> MethodResultSerializer</code> class serializes a
     * <code>{@link MethodResult}</code> into a JSON object value.
     */
    @SuppressWarnings("serial")
    private static class MethodResultSerializer extends StdSerializer<MethodResult> {
        /** Constructor. */
        public MethodResultSerializer() {
            super(MethodResult.class);
        }

        /**
         * Serializes the provided <code>{@link MethodResult}</code> into a
         * JSON object value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param methodResult     MethodResult to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(MethodResult methodResult,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            DataValue output = methodResult.getOutput();
            ErrorValue error = methodResult.getError();
            jgen.writeStartObject();
            if (null != output) {
                provider.defaultSerializeField(METHOD_RESULT_OUTPUT, output, jgen);
            } else if (null != error) {
                provider.defaultSerializeField(METHOD_RESULT_ERROR, error, jgen);
            }
            jgen.writeEndObject();
        }
    }

    /**
     * The <code> ExecutionContextSerializer</code> class serializes a
     * <code>{@link ExecutionContext}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class AppContextSerializer extends StdSerializer<ExecutionContext> {
        private static final String APP_CTX = "appCtx";

        /** Constructor **/
        public AppContextSerializer() {
            super(ExecutionContext.class);
        }

        /**
         * Serializes the provided <code>{@link ExecutionContext}</code> into a
         * JSON object value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param ctx       ExecutionContext to serialize
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(ExecutionContext ctx,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();
            ApplicationData additionalRequestData = ctx.retrieveApplicationData();
            if (additionalRequestData != null) {
                provider.defaultSerializeField(APP_CTX,
                                               additionalRequestData.getAllProperties(), jgen);
            } else {
                provider.defaultSerializeField(APP_CTX,
                                               Collections.<String, String>emptyMap(), jgen);
            }
            jgen.writeEndObject();
        }
    }

    /**
     * The <code>ApiRequestSerializer</code> class serializes a
     * <code>{@link JsonApiRequest}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class ApiRequestSerializer extends StdSerializer<JsonApiRequest> {
        /** Constructor **/
        public ApiRequestSerializer() {
            super(JsonApiRequest.class);
        }

        /**
         * Serializes the provided <code>{@link JsonApiRequest}</code> into a
         * JSON object value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param request   request data to be serialized
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(JsonApiRequest request,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();   // start request
            jgen.writeStringField(FIELD_JSONRPC, JSON_RPC_VERSION);
            jgen.writeStringField(FIELD_ID, request.getId());
            jgen.writeStringField(FIELD_METHOD, request.getMethod());

            JsonInvokeRequestParams2 params =
                (JsonInvokeRequestParams2) request.getParams();
            jgen.writeFieldName(FIELD_PARAMS);
            jgen.writeStartObject();   // start "params" value
            jgen.writeStringField(FIELD_SERVICE_ID, params.getServiceId());
            jgen.writeStringField(FIELD_OPERATION_ID, params.getOperationId());

            jgen.writeObjectField(CONTEXT, params.getCtx());
            jgen.writeObjectField(INPUT, params.getInput());
            jgen.writeEndObject();     // end "params" value

            jgen.writeEndObject();    // end request
        }
    }

    /**
     * The <code>ApiRequestSerializer</code> class serializes a
     * <code>{@link JsonApiRequest}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class ApiResponseSerializer extends StdSerializer<JsonApiResponse> {
        /** Constructor **/
        public ApiResponseSerializer() {
            super(JsonApiResponse.class);
        }

        /**
         * Serializes the provided <code>{@link JsonApiResponse}</code> into a
         * JSON object value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param response  response data to be serialized
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(JsonApiResponse response,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();   // start request
            jgen.writeStringField(FIELD_JSONRPC, JSON_RPC_VERSION);
            jgen.writeStringField(FIELD_ID, response.getId());
            jgen.writeObjectField(FIELD_RESULT, response.getResult());
        }
    }

    /**
     * The <code>JsonErrorSerializer</code> class serializes a
     * <code>{@link JsonError}</code> into JSON.
     */
    @SuppressWarnings("serial")
    private static class JsonErrorSerializer extends StdSerializer<JsonError> {
        /** Constructor **/
        public JsonErrorSerializer() {
            super(JsonError.class);
        }

        /**
         * Serializes the provided {@link JsonError}</code> into a
         * JSON object value and writes the result using the provided
         * <code>{@link JsonGenerator}</code>.
         *
         * @param error     error to be serialized
         * @param jgen      generator used to output resulting JSON content
         * @param provider  provider that can be used to get serializers for
         *                  sub-objects
         *
         * @throws  IOException if processing or writing the JSON value fails, might be
         * {@link JsonProcessingException}
         */
        @Override
        public void serialize(JsonError error,
                              JsonGenerator jgen,
                              SerializerProvider provider) throws IOException {
            jgen.writeStartObject();   // { - start error
            jgen.writeNumberField(JSON_ERROR_CODE, error.getCode());
            jgen.writeStringField(JSON_ERROR_MESSAGE, error.getMessage());
            if (error.getData() != null) {
                jgen.writeStringField(JSON_ERROR_DATA, error.getData());
            }
            jgen.writeEndObject();    // }
        }
    }

    @Override
    public byte[] serialize(Object object) {
        try {
            return mapper.writeValueAsBytes(object);
        } catch (IOException e) {
            logger.error("Failed to serialize JSON request", e);
            throw toVapiCoreException(e);
        }
    }

    @Override
    public String serializeDataValue(DataValue dataValue) {
        return serializeToString(dataValue);
    }

    public String serializeToString(Object object) {
        ObjectMapper mapper = new ObjectMapper();
        mapper.registerModule(module);
        try {
            return mapper.writeValueAsString(object);
        } catch (IOException e) {
            logger.error("Failed to serialize JSON request", e);
            throw toVapiCoreException(e);
        }
    }

    /**
     * Converts an Exception into a {@link CoreException}.
     *
     * @param ex  exception
     * @return    vAPI exception that wraps the provided exception
     */
    private static CoreException toVapiCoreException(Exception ex) {
        return new CoreException(
                MessageFactory.getMessage("vapi.json.serialize.ioerror",
                                          ex.getMessage()),
                ex);
    }
}
