/* **********************************************************
 * Copyright (c) 2012-2013, 2016, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.common.json;

import java.io.IOException;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.dsig.json.SignatureException;
import com.vmware.vapi.internal.util.Validate;

/**
 * This class is responsible for serialization/deserialization of the
 * SecurityContext wire structure in JSON request messages.
 */
public final class JsonSecurityContextSerializer {

    // TODO unit test the class

    // the request default validity window in minutes
    public static final int TS_DEF_OFFSET = 10; // minutes
    private static final Message SERIALIZATION_ERROR = MessageFactory
            .getMessage("vapi.signature.serialization");
    private static final String SEC_CTX_NAME = "securityCtx";
    private static final Pattern EXEC_CTX_REGEXP = Pattern.compile("\"ctx\"\\s*:\\s*\\{");
    private static final String JSON_OBJ_DELIM = ":";
    private static final String QUOT = "\"";
    private static final ObjectMapper jsonMapper = new ObjectMapper();

    public JsonSecurityContextSerializer() {
    }

    /**
     * Serializes the SecurityContext wire structure into the given JSON request
     *
     * @param securityContext required
     * @param request required
     * @return the modified request that contains the SecurityContext structure
     */
    public String serializeSecurityContext(Map<String, Object> securityContext, String request) {
        return serializeSecurityContext(toJsonString(securityContext), request);
    }

    /**
     * Removes the signature from the security context.
     *
     * @param request the JSON request. must not be null.
     * @return the modified JSON request that will not contain signature in its
     *         security context.
     */
    public String removeSignature(String request) {
        Validate.notNull(request);
        // TODO this is slow! refactor later.
        JsonNode requestTree = parseToTree(request);
        JsonNode secCtx = requestTree.findPath(SEC_CTX_NAME);
        ((ObjectNode)secCtx).remove("signature");
        return toJsonString(requestTree);
    }

    /**
     * Serializes the SecurityContext into the JSON request
     *
     * @param securityContext required.
     * @param request required.
     * @return the JSON request with SecurityContext.
     */
    private String serializeSecurityContext(String securityContext, String request) {
        Validate.notNull(securityContext);
        Validate.notNull(request);

        StringBuilder result = new StringBuilder();
        // TODO what if exec ctx is missing
        int pos = findEndPosition(request, EXEC_CTX_REGEXP);
        // TODO what if the execution context is empty
        StringBuilder secCtx = new StringBuilder();
        quote(secCtx, SEC_CTX_NAME).append(JSON_OBJ_DELIM);
        secCtx.append(securityContext);
        result.append(request.substring(0, pos)).append(secCtx.toString()).append(",").append(request.substring(pos));
        return result.toString();
    }

    /**
     * @param message
     * @return parses the message to a JSON tree structure. If the message is
     *         <code>null</code> the return value will be also <code>null</code>
     */
    private static JsonNode parseToTree(String message) {
        if (message == null) {
            return null;
        }

        JsonNode root;
        try {
            root = jsonMapper.readTree(message);
        } catch (IOException e) {
            throw new SignatureException(SERIALIZATION_ERROR, e);
        }
        return root;
    }

    /**
     * Quotes a the given value.
     *
     * @param builder required.
     * @param value required.
     * @return
     */
    private StringBuilder quote(StringBuilder builder, String value) {
        assert builder != null;
        assert value != null;

        return builder.append(QUOT).append(value).append(QUOT);
    }

    // TODO private??
    /**
     * Serializes the given object to a JSON string.
     *
     * @param ctx required.
     * @return the resulting JSON string.
     */
    public String toJsonString(Object ctx) {
        try {
            return jsonMapper.writeValueAsString(ctx);
        } catch (IOException e) {
            throw new SignatureException(SERIALIZATION_ERROR, e);
        }
    }

    /**
     * @param request required.
     * @param regexp required.
     * @return the ending char index of the first occurrence of the regexp. If the
     *         regexp is not found -1 will be returned.
     */
    private int findEndPosition(String request, Pattern regexp) {
        assert request != null;
        assert regexp != null;

        Matcher m = regexp.matcher(request);

        if (m.find()) {
            return m.end();
        }

        return -1;
    }
}
