/* **********************************************************
 * Copyright (c) 2013-2014, 2016, 2020 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.common.msg;

import java.io.IOException;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.vmware.vapi.CoreException;
import com.vmware.vapi.client.exception.InternalClientException;
import com.vmware.vapi.client.exception.MessageProtocolException;
import com.vmware.vapi.client.exception.TransportProtocolException;
import com.vmware.vapi.internal.common.exception.ExceptionTranslatorBase;
import com.vmware.vapi.internal.protocol.common.json.JsonInvalidDataValueException;
import com.vmware.vapi.internal.protocol.common.json.JsonInvalidMethodException;
import com.vmware.vapi.internal.protocol.common.json.JsonInvalidMethodParamsException;

public class JsonMessageProtocolExceptionTranslator extends ExceptionTranslatorBase {
    /**
     * Translate 3rd party exception to our own exception hierarchy of client
     * exceptions. The exception stacktrace is fixed as if it was created in the
     * calling method.
     *
     * @param e the exception to be translated
     * @return the translated exception
     */
    public static RuntimeException translate(Exception e) {
        if (e instanceof JsonInvalidDataValueException ||
            e instanceof JsonInvalidMethodException ||
            e instanceof JsonInvalidMethodParamsException ||
            e instanceof JsonProcessingException ||
            e instanceof CoreException) {
            return fixStackTrace(new MessageProtocolException(e.getMessage(), e));
        } else if (e instanceof IOException) {
            // If the JSON deserializer fails while reading the input stream we can
            // have a transport layer failure here, so:
            return fixStackTrace(new TransportProtocolException(e.getMessage(), e));
        }

        // Unknown exception, don't translate
        if (e instanceof RuntimeException) {
            return (RuntimeException) e;
        }

        return new InternalClientException(e.getMessage(), e);
    }
}