/* **********************************************************
 * Copyright 2011-2015, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.protocol.server.rpc.http.util;

import java.io.File;

import com.vmware.vapi.internal.util.Validate;

/**
 * Common utilities to manage files
 */
public final class FileUtil {

   static final String CLASSPATH_URL_PREFIX = "classpath:";
   static final String FILE_SYSTEM_URL_PREFIX = "file:";

   private FileUtil() {
      // prevent instantiation
   }

   /**
    * Returns an absolute file path from an absolute/relative one. If
    * <code>path</code> starts with "classpath:", it is searched in the
    * classpath. If <code>path</code> starts with "file:", it refers to a file
    * system path (relative or absolute) - this is for consistency with spring
    * resource. Otherwise, <code>path</code> refers to file system path
    * (relative or absolute).
    * 
    * @param path relative filename; must not be <code>null</code>
    * @return absolute filename
    */
   public static String getAbsoluteFilename(String path) {
      Validate.notNull(path);
      if (path.startsWith(CLASSPATH_URL_PREFIX)) {
         // Get from classpath if prefixed
         ClassLoader loader = FileUtil.class.getClassLoader();
         return loader.getResource(stripPrefix(path)).getPath();
      } else if (path.startsWith(FILE_SYSTEM_URL_PREFIX)) {
         path = stripPrefix(path);
      }
      // Load from filesytem
      return new File(path).getAbsolutePath();
   }

   private static String stripPrefix(String path) {
      int indexOfColon = path.indexOf(':');
      if (indexOfColon >= 0) {
         return path.substring(indexOfColon + 1);
      } else {
         return path;
      }
   }
}
