/* **********************************************************
 * Copyright (c) 2013-2014, 2016 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.provider.introspection;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Map;
import java.util.Set;

import com.vmware.vapi.core.InterfaceDefinition;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.core.MethodDefinition;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.core.ProviderDefinition;
import com.vmware.vapi.data.AnyErrorDefinition;
import com.vmware.vapi.data.BlobDefinition;
import com.vmware.vapi.data.BooleanDefinition;
import com.vmware.vapi.data.DataDefinition;
import com.vmware.vapi.data.DataType;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.DefinitionVisitor;
import com.vmware.vapi.data.DoubleDefinition;
import com.vmware.vapi.data.DynamicStructDefinition;
import com.vmware.vapi.data.ErrorDefinition;
import com.vmware.vapi.data.IntegerDefinition;
import com.vmware.vapi.data.ListDefinition;
import com.vmware.vapi.data.ListValue;
import com.vmware.vapi.data.OpaqueDefinition;
import com.vmware.vapi.data.OptionalDefinition;
import com.vmware.vapi.data.OptionalValue;
import com.vmware.vapi.data.SecretDefinition;
import com.vmware.vapi.data.StringDefinition;
import com.vmware.vapi.data.StringValue;
import com.vmware.vapi.data.StructDefinition;
import com.vmware.vapi.data.StructRefDefinition;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.data.VoidDefinition;
import com.vmware.vapi.internal.data.ReferenceResolver;
import com.vmware.vapi.provider.introspection.IntrospectionConstants;
import com.vmware.vapi.std.BuiltInDataFactory;
import com.vmware.vapi.std.StandardDataFactory;

/**
 * Factory for {@link DataDefinition}s and {@link DataValue}s for
 * standard vAPI structures used in <tt>Introspection</tt> API.
 */
public final class IntrospectionDataFactory {

    // operation identifiers
    public final static MethodIdentifier PROVIDER_GET_METHOD_ID =
            new MethodIdentifier(
                    IntrospectionConstants.PROVIDER_INTROSPECTION_SERVICE_ID,
                    "get");
    public final static MethodIdentifier SERVICE_LIST_METHOD_ID =
            new MethodIdentifier(
                    IntrospectionConstants.SERVICE_INTROSPECTION_SERVICE_ID,
                    "list");
    public final static MethodIdentifier SERVICE_GET_METHOD_ID =
            new MethodIdentifier(
                    IntrospectionConstants.SERVICE_INTROSPECTION_SERVICE_ID,
                    "get");
    public final static MethodIdentifier OPERATION_GET_METHOD_ID =
            new MethodIdentifier(
                    IntrospectionConstants.OPERATION_INTROSPECTION_SERVICE_ID,
                    "get");
    public final static MethodIdentifier OPERATION_LIST_METHOD_ID =
            new MethodIdentifier(
                    IntrospectionConstants.OPERATION_INTROSPECTION_SERVICE_ID,
                    "list");

    // operation parameters
    public final static String SERVICE_GET_METHOD_ID_PARAM = "id";
    public final static String OPERATION_GET_METHOD_SERVICE_ID_PARAM =
            "service_id";
    public final static String OPERATION_GET_METHOD_OPERATION_ID_PARAM =
            "operation_id";
    public final static String OPERATION_LIST_METHOD_SERVICE_ID_PARAM =
            "service_id";

    // structure names
    public static final String PROVIDER_INFO_NAME =
        "com.vmware.vapi.std.introspection.provider.info";
    public static final String SERVICE_INFO_NAME =
        "com.vmware.vapi.std.introspection.service.info";
    public static final String OPERATION_INFO_NAME =
        "com.vmware.vapi.std.introspection.operation.info";
    public static final String OPERATION_DATA_DEF_NAME =
        "com.vmware.vapi.std.introspection.operation.data_definition";
    public static final String DATA_DEFINITION_NAME =
        "com.vmware.vapi.std.introspection.operation.data_definition";
    public static final String DATA_DEF_FIELDS_MAP_ENTRY_NAME =
            BuiltInDataFactory.MAP_ENTRY_STRUCT_NAME;


    // structure field names
    public static final String PROVIDER_INFO_FIELD_ID = "id";
    public static final String PROVIDER_INFO_FIELD_CHECKSUM = "checksum";

    public static final String SERVICE_INFO_FIELD_OPERATIONS = "operations";

    public static final String OPERATION_INFO_FIELD_INPUTDEF = "input_definition";
    public static final String OPERATION_INFO_FIELD_OUTPUTDEF = "output_definition";
    public static final String OPERATION_INFO_FIELD_ERRORDEF = "error_definitions";

    public static final String DATA_DEFINITION_FIELD_TYPE = "type";
    public static final String DATA_DEFINITION_FIELD_ELEMENT_DEF = "element_definition";
    public static final String DATA_DEFINITION_FIELD_NAME = "name";
    public static final String DATA_DEFINITION_FIELD_FIELDS = "fields";

    // map entry field names
    public static final String MAP_KEY_FIELD = "key";
    public static final String MAP_VALUE_FIELD = "value";

    // structure definitions
    public static final StructDefinition PROVIDER_INFO_DEF;
    public static final StructDefinition SERVICE_INFO_DEF;
    public static final StructDefinition OPERATION_INFO_DEF;
    public static final StructDefinition OPERATION_DATA_DEF_DEF;
    public static final StructDefinition DATA_DEFINITION_FIELDS_MAP_DEF;

    // error definitions
    public static final ErrorDefinition NOT_FOUND_ERROR_INFO_DEF;

    // data definition type names
    public static final String DATA_TYPE_VOID = "VOID";
    public static final String DATA_TYPE_LONG = "LONG";
    public static final String DATA_TYPE_DOUBLE = "DOUBLE";
    public static final String DATA_TYPE_BOOLEAN = "BOOLEAN";
    public static final String DATA_TYPE_BINARY = "BINARY";
    public static final String DATA_TYPE_STRING = "STRING";
    public static final String DATA_TYPE_ERROR = "ERROR";
    public static final String DATA_TYPE_OPTIONAL = "OPTIONAL";
    public static final String DATA_TYPE_LIST = "LIST";
    public static final String DATA_TYPE_STRUCTURE = "STRUCTURE";
    public static final String DATA_TYPE_OPAQUE = "OPAQUE";
    public static final String DATA_TYPE_SECRET = "SECRET";
    public static final String DATA_TYPE_STRUCTURE_REF = "STRUCTURE_REF";
    public static final String DATA_TYPE_DYNAMIC_STRUCTURE = "DYNAMIC_STRUCTURE";
    public static final String DATA_TYPE_ANY_ERROR = "ANY_ERROR";

    static {
        Map<String, DataDefinition> fields;

        // Provider.Info def
        fields = new HashMap<String, DataDefinition>();
        fields.put(PROVIDER_INFO_FIELD_ID, StringDefinition.getInstance());
        fields.put(PROVIDER_INFO_FIELD_CHECKSUM, StringDefinition.getInstance());
        PROVIDER_INFO_DEF = new StructDefinition(PROVIDER_INFO_NAME, fields);

        // Service.Info def
        fields = Collections.singletonMap(SERVICE_INFO_FIELD_OPERATIONS,
            (DataDefinition) new ListDefinition(StringDefinition.getInstance()));
        SERVICE_INFO_DEF = new StructDefinition(SERVICE_INFO_NAME, fields);

        // Operation.Info def
        StructRefDefinition OPERATION_DATA_DEF_REF_DEF = new StructRefDefinition(OPERATION_DATA_DEF_NAME);

        fields = new HashMap<>();
        fields.put(MAP_KEY_FIELD, StringDefinition.getInstance());
        fields.put(MAP_VALUE_FIELD, OPERATION_DATA_DEF_REF_DEF);
        DATA_DEFINITION_FIELDS_MAP_DEF = new StructDefinition(
                DATA_DEF_FIELDS_MAP_ENTRY_NAME,
                fields);

        fields = new HashMap<>();
        fields.put("type", StringDefinition.getInstance());
        fields.put("element_definition", new OptionalDefinition(OPERATION_DATA_DEF_REF_DEF));
        fields.put("name", new OptionalDefinition(StringDefinition.getInstance()));
        fields.put("fields", new OptionalDefinition(new ListDefinition(DATA_DEFINITION_FIELDS_MAP_DEF)));
        OPERATION_DATA_DEF_DEF = new StructDefinition(OPERATION_DATA_DEF_NAME, fields);

        // resolve the StructRefDefinition
        OPERATION_DATA_DEF_REF_DEF.setTarget(OPERATION_DATA_DEF_DEF);

        fields = new HashMap<>();
        fields.put(OPERATION_INFO_FIELD_INPUTDEF, OPERATION_DATA_DEF_DEF);
        fields.put(OPERATION_INFO_FIELD_OUTPUTDEF, OPERATION_DATA_DEF_DEF);
        fields.put(OPERATION_INFO_FIELD_ERRORDEF, new ListDefinition(OPERATION_DATA_DEF_DEF));
        OPERATION_INFO_DEF = new StructDefinition(OPERATION_INFO_NAME, fields);

        // NotFound error def
        NOT_FOUND_ERROR_INFO_DEF = StandardDataFactory.createStandardErrorDefinition(
                "com.vmware.vapi.std.errors.not_found");
    }

    public static DataValue toProviderInfo(ProviderDefinition provider) {
        StructValue result = PROVIDER_INFO_DEF.newInstance();
        result.setField(PROVIDER_INFO_FIELD_ID, provider.getIdentifier());
        result.setField(PROVIDER_INFO_FIELD_CHECKSUM, provider.getCheckSum());

        return result;
    }

    public static ListValue toServiceIdList(
            Set<InterfaceIdentifier> ifaceIds) {

        ListValue list = new ListValue();
        for (InterfaceIdentifier iId : ifaceIds) {
            list.add(new StringValue(iId.getName()));
        }
        return list;
    }

    public static DataValue toServiceInfo(InterfaceDefinition iface) {
        StructValue result = SERVICE_INFO_DEF.newInstance();

        ListValue operations = toMethodIdList(iface);
        result.setField(SERVICE_INFO_FIELD_OPERATIONS, operations);
        return result;
    }

    public static ListValue toMethodIdList(InterfaceDefinition iface) {
        ListValue operations = new ListValue();
        for (MethodIdentifier methodId : iface.getMethodIdentifiers()) {
            operations.add(new StringValue(methodId.getName()));
        }
        return operations;
    }

    public static StructValue toOperationInfo(MethodDefinition method) {
        StructValue result = OPERATION_INFO_DEF.newInstance();

        result.setField(OPERATION_INFO_FIELD_INPUTDEF,
                        toDataDefinition(method.getInputDefinition()));
        result.setField(OPERATION_INFO_FIELD_OUTPUTDEF,
                        toDataDefinition(method.getOutputDefinition()));
        result.setField(OPERATION_INFO_FIELD_ERRORDEF,
                        toDataDefinitionList(method.getErrorDefinitions()));

        return result;
    }

    public static DataValue toDataDefinition(DataDefinition dataDef) {
        DefinitionVisitorImpl visitor = new DefinitionVisitorImpl();
        dataDef.accept(visitor);
        return visitor.getValue();
    }

    public static ListValue toDataDefinitionList(
            Set<? extends DataDefinition> dataDefs) {

        ListValue result = new ListValue();
        for (DataDefinition def : dataDefs) {
            result.add(toDataDefinition(def));
        }
        return result;
    }

    private static class DefinitionVisitorImpl implements DefinitionVisitor {

        private DataValue value;

        public DataValue getValue() {
            return value;
        }

        // methods below - only "type" needed
        @Override
        public void visit(VoidDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(BooleanDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(IntegerDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(DoubleDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(StringDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(BlobDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(OpaqueDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(SecretDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(DynamicStructDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        @Override
        public void visit(AnyErrorDefinition def) {
            value = buildDataDefWithType(def.getType());
        }

        // methods below - "type" and "element_definition" needed
        @Override
        public void visit(OptionalDefinition def) {
            value = buildDataDefWithElement(def.getType(), def.getElementType());
        }

        @Override
        public void visit(ListDefinition def) {
            value = buildDataDefWithElement(def.getType(), def.getElementType());
        }


        // methods below - "type" and "name"
        @Override
        public void visit(StructRefDefinition def) {
            value = buildDataDefWithName(def.getType(), def.getName());
        }

        // methods below - "type", "name" and "fields"
        @Override
        public void visit(StructDefinition def) {
            value = buildDataDefWithFields(def);
        }

        private static StructValue buildDataDefWithType(DataType type) {
            StructValue struct = new StructValue(DATA_DEFINITION_NAME);
            struct.setField(DATA_DEFINITION_FIELD_TYPE, typeToString(type));
            struct.setField(DATA_DEFINITION_FIELD_NAME, OptionalValue.UNSET);
            struct.setField(DATA_DEFINITION_FIELD_ELEMENT_DEF, OptionalValue.UNSET);
            struct.setField(DATA_DEFINITION_FIELD_FIELDS, OptionalValue.UNSET);
            return struct;
        }

        private static StructValue buildDataDefWithName(DataType type, String name) {
            StructValue struct = buildDataDefWithType(type);
            struct.setField(DATA_DEFINITION_FIELD_NAME, new OptionalValue(name));
            return struct;
        }

        private static StructValue buildDataDefWithElement(
                DataType type, DataDefinition elementDef) {
            StructValue struct = buildDataDefWithType(type);
            DataValue elementDefValue = toDataDefinition(elementDef);
            struct.setField(DATA_DEFINITION_FIELD_ELEMENT_DEF,
                            new OptionalValue(elementDefValue));
            return struct;
        }

        private static StructValue buildDataDefWithFields(StructDefinition def) {
            StructValue struct = buildDataDefWithName(def.getType(), def.getName());
            ListValue list = new ListValue();
            for (String fieldName : def.getFieldNames()) {
                DataDefinition fieldDef = def.getField(fieldName);
                StructValue mapEntryStruct =
                    DATA_DEFINITION_FIELDS_MAP_DEF.newInstance();
                mapEntryStruct.setField(MAP_KEY_FIELD, fieldName);
                mapEntryStruct.setField(MAP_VALUE_FIELD, toDataDefinition(fieldDef));
                list.add(mapEntryStruct);
            }
            struct.setField(DATA_DEFINITION_FIELD_FIELDS,
                            new OptionalValue(list));
            return struct;
        }
    }

    private static String typeToString(DataType type) {
        switch (type) {
            case VOID: return DATA_TYPE_VOID;
            case INTEGER: return DATA_TYPE_LONG;
            case DOUBLE: return DATA_TYPE_DOUBLE;
            case BOOLEAN: return DATA_TYPE_BOOLEAN;
            case BLOB: return DATA_TYPE_BINARY;
            case STRING: return DATA_TYPE_STRING;
            case ERROR: return DATA_TYPE_ERROR;
            case OPTIONAL: return DATA_TYPE_OPTIONAL;
            case LIST: return DATA_TYPE_LIST;
            case STRUCTURE: return DATA_TYPE_STRUCTURE;
            case OPAQUE: return DATA_TYPE_OPAQUE;
            case SECRET: return DATA_TYPE_SECRET;
            case STRUCTURE_REF: return DATA_TYPE_STRUCTURE_REF;
            case DYNAMIC_STRUCTURE: return DATA_TYPE_DYNAMIC_STRUCTURE;
            case ANY_ERROR: return DATA_TYPE_ANY_ERROR;
            default:
                throw new IntrospectionException(getMessage(
                        "vapi.introspection.data.unknown.data.type.enum",
                        type.toString()));
        }
    }

    public static StructValue createMethodInput() {
        return new StructValue(
                BuiltInDataFactory.OPERATION_INPUT_STRUCT_NAME);
    }

    public static ProviderDefinition fromProviderInfo(DataValue value) {
        try {
            StructValue struct = toStruct(value, PROVIDER_INFO_NAME);
            String id = struct.getString(PROVIDER_INFO_FIELD_ID);
            String checkSum = struct.getString(PROVIDER_INFO_FIELD_CHECKSUM);
            return new ProviderDefinition(id, checkSum);
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.provider.info.error"),
                    ex);
        }
    }

    /**
     * Converts data definition value to binding. The structure references in
     * the returned object must be manually resolved!
     *
     * @param value data definition as value
     * @param referenceResolver resolver for structure references
     * @return data definition binding object
     * @throws IntrospectionException
     */
    static DataDefinition fromDataDefinition(DataValue value,
            ReferenceResolver referenceResolver) {
        try {
            return fromDataDefinitionImpl(value, referenceResolver);
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.def.error"),
                    ex);
        }
    }

    /**
     * Same as {@link #fromDataDefinition(DataValue)} without exception
     * translation.
     */
    private static DataDefinition fromDataDefinitionImpl(DataValue value,
            ReferenceResolver referenceResolver) {
        StructValue struct = toStruct(value, DATA_DEFINITION_NAME);
        String typeName = struct.getString(DATA_DEFINITION_FIELD_TYPE);

        if (DATA_TYPE_VOID.equals(typeName)) {
            return VoidDefinition.getInstance();
        }

        if (DATA_TYPE_LONG.equals(typeName)) {
            return IntegerDefinition.getInstance();
        }

        if (DATA_TYPE_DOUBLE.equals(typeName)) {
            return DoubleDefinition.getInstance();
        }

        if (DATA_TYPE_BOOLEAN.equals(typeName)) {
            return BooleanDefinition.getInstance();
        }

        if (DATA_TYPE_BINARY.equals(typeName)) {
            return BlobDefinition.getInstance();
        }

        if (DATA_TYPE_STRING.equals(typeName)) {
            return StringDefinition.getInstance();
        }

        if (DATA_TYPE_OPAQUE.equals(typeName)) {
            return OpaqueDefinition.getInstance();
        }

        if (DATA_TYPE_SECRET.equals(typeName)) {
            return SecretDefinition.getInstance();
        }

        if (DATA_TYPE_DYNAMIC_STRUCTURE.equals(typeName)) {
            return DynamicStructDefinition.getInstance();
        }

        if (DATA_TYPE_ANY_ERROR.equals(typeName)) {
            return AnyErrorDefinition.getInstance();
        }

        if (DATA_TYPE_OPTIONAL.equals(typeName)) {
            return new OptionalDefinition(getElementDefinition(struct,
                    referenceResolver));
        }

        if (DATA_TYPE_LIST.equals(typeName)) {
            return new ListDefinition(getElementDefinition(struct,
                    referenceResolver));
        }

        boolean isError = DATA_TYPE_ERROR.equals(typeName);
        if (isError || DATA_TYPE_STRUCTURE.equals(typeName)) {
            String name = getDataDefinitionName(struct);
            Map<String, DataDefinition> fields = getDataDefinitionFields(
                    struct, typeName, referenceResolver);
            if (isError) {
                return new ErrorDefinition(name, fields);
            } else {
                StructDefinition d = new StructDefinition(name, fields);
                referenceResolver.addDefinition(d);
                return d;
            }
        }

        if (DATA_TYPE_STRUCTURE_REF.equals(typeName)) {
            StructRefDefinition r = new StructRefDefinition(
                    getDataDefinitionName(struct));
            referenceResolver.addReference(r);
            return r;
        }

        throw new IntrospectionException(getMessage(
                "vapi.introspection.data.unknown.data.type.value", typeName));
    }

    private static String getDataDefinitionName(StructValue struct) {
        try {
            return struct.getOptional(DATA_DEFINITION_FIELD_NAME).getString();
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.def.name.error"), ex);
        }
    }

    private static Map<String, DataDefinition> getDataDefinitionFields(
            StructValue struct, String typeName,
            ReferenceResolver referenceResolver) {
        ListValue list;
        try {
            list = struct.getOptional(DATA_DEFINITION_FIELD_FIELDS).getList();
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.def.fields.error"), ex);
        }
        Map<String, DataDefinition> fields =
                new HashMap<String, DataDefinition>();

        for (DataValue e : list) {
            addDataDefinitionFieldsEntry(fields, e, referenceResolver);
        }
        return fields;
    }

    private static void addDataDefinitionFieldsEntry(
            Map<String, DataDefinition> fields,
            DataValue value,
            ReferenceResolver referenceResolver) {
        try {
            StructValue mapEntryStruct =
                    toStruct(value, DATA_DEF_FIELDS_MAP_ENTRY_NAME);
            String key = mapEntryStruct.getString(MAP_KEY_FIELD);
            StructValue fieldDefStruct =
                    mapEntryStruct.getStruct(MAP_VALUE_FIELD);
            DataDefinition fieldDef =
                    fromDataDefinition(fieldDefStruct, referenceResolver);
            fields.put(key, fieldDef);
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.def.fields.entry.error"),
                    ex);
        }
    }

    private static DataDefinition getElementDefinition(StructValue dataDef,
            ReferenceResolver referenceResolver) {
        try {
            return fromDataDefinition(
                    dataDef.getOptional(DATA_DEFINITION_FIELD_ELEMENT_DEF)
                            .getStruct(), referenceResolver);
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.def.element.error"), ex);
        }
    }

    public static MethodDefinition fromOperationInfo(DataValue value,
            MethodIdentifier methodId) {
        try {
            return fromOperationInfoImpl(value, methodId);
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.operation.info.error"),
                    ex);
        }
    }

    /**
     * Same as {@link #fromOperationInfo(DataValue, MethodIdentifier)} but
     * without exception translation.
     */
    private static MethodDefinition fromOperationInfoImpl(DataValue value,
            MethodIdentifier methodId) {
        StructValue struct = toStruct(value, OPERATION_INFO_NAME);
        DataDefinition inputDef = getOperationInfoDefinitionField(struct,
                OPERATION_INFO_FIELD_INPUTDEF);
        DataDefinition outputDef = getOperationInfoDefinitionField(struct,
                OPERATION_INFO_FIELD_OUTPUTDEF);
        ListValue list = getErrorsFromOperationInfo(struct);
        Set<ErrorDefinition> errors = new HashSet<ErrorDefinition>();
        for (DataValue e : list) {
            errors.add(fromErrorDefinition(e));
        }
        return new MethodDefinition(methodId, inputDef, outputDef, errors);
    }

    private static DataDefinition getOperationInfoDefinitionField(
            StructValue opInfo,
            String fieldName) {
        try {
            ReferenceResolver resolver = new ReferenceResolver();
            DataDefinition def = fromDataDefinition(opInfo.getField(fieldName),
                    resolver);
            resolver.resolveReferences();
            return def;
        } catch (RuntimeException ex) {
            throw new IntrospectionException(getMessage(
                    "vapi.introspection.data.operation.info.def.error",
                    fieldName), ex);
        }
    }

    private static ErrorDefinition fromErrorDefinition(
            DataValue value) {
        try {
            StructValue struct = toStruct(value, DATA_DEFINITION_NAME);
            String typeName = struct.getString(DATA_DEFINITION_FIELD_TYPE);
            if (!DATA_TYPE_ERROR.equals(typeName)) {
                throw new IntrospectionException(
                        getMessage(
                                "vapi.introspection.data.operation.info.error.def.invalid.type",
                                typeName));
            }
            ReferenceResolver resolver = new ReferenceResolver();
            String name = getDataDefinitionName(struct);
            Map<String, DataDefinition> fields = getDataDefinitionFields(
                    struct, typeName, resolver);
            ErrorDefinition errorDef = new ErrorDefinition(name, fields);
            resolver.resolveReferences();
            return errorDef;
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.operation.info.error.def.error"),
                    ex);
        }
    }

    private static ListValue getErrorsFromOperationInfo(
            StructValue operationInfo) {
        return operationInfo.getList(OPERATION_INFO_FIELD_ERRORDEF);
    }

    private static MethodIdentifier fromOperationId(DataValue value,
            InterfaceIdentifier ifaceId) {
        if (!(value instanceof StringValue)) {
            throw new IntrospectionException(getMessage(
                    "vapi.introspection.data.operation.id.error",
                    getValueName(value)));
        }
        StringValue methodName = (StringValue) value;
        return new MethodIdentifier(ifaceId, methodName.getValue());
    }

    public static InterfaceDefinition fromServiceInfo(DataValue value,
            InterfaceIdentifier ifaceId) {
        try {
            Set<MethodIdentifier> methods = new HashSet<MethodIdentifier>();
            StructValue struct = toStruct(value, SERVICE_INFO_NAME);
            ListValue list = struct.getList(SERVICE_INFO_FIELD_OPERATIONS);
            for (DataValue e : list) {
                methods.add(fromOperationId(e, ifaceId));
            }
            return new InterfaceDefinition(ifaceId, methods);
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.service.info.error"),
                    ex);
        }
    }

    private static InterfaceIdentifier fromServiceId(DataValue value) {
        if (!(value instanceof StringValue)) {
            throw new IntrospectionException(getMessage(
                    "vapi.introspection.data.service.id.error",
                    getValueName(value)));
        }
        StringValue s = (StringValue) value;
        return new InterfaceIdentifier(s.getValue());
    }

    public static Set<InterfaceIdentifier> fromServiceIdList(DataValue value) {
        if (!(value instanceof ListValue)) {
            throw new IntrospectionException(getMessage(
                    "vapi.introspection.data.service.id.list.error",
                    getValueName(value)));
        }
        ListValue list = (ListValue) value;
        Set<InterfaceIdentifier> services = new HashSet<InterfaceIdentifier>();
        for (DataValue e : list) {
            services.add(fromServiceId(e));
        }
        return services;
    }

    public static void addErrorsToOperationInfo(DataValue operationInfo,
            Iterable<? extends DataValue> extraErrors) {
        try {
            addErrorsToOperationInfoImpl(operationInfo, extraErrors);
        } catch (RuntimeException ex) {
            throw new IntrospectionException(
                    getMessage("vapi.introspection.data.operation.info.error.def.add.error"),
                    ex);
        }
    }

    /**
     * Same as {@link #addErrorsToOperationInfo(DataValue, Iterable)} but
     * without exception translation.
     */
    private static void addErrorsToOperationInfoImpl(DataValue operationInfo,
            Iterable<? extends DataValue> extraErrors) {
        StructValue struct = toStruct(operationInfo, OPERATION_INFO_NAME);
        ListValue errorList = getErrorsFromOperationInfo(struct);
        Set<String> errorNames = getErrorNames(errorList);
        for (DataValue e : extraErrors) {
            StructValue errorStruct = toStruct(e, DATA_DEFINITION_NAME);
            String name = getDataDefinitionName(errorStruct);
            // add an extra error only if it does not exist
            if (!errorNames.contains(name)) {
                errorList.add(e);
            }
        }
    }

    private static Set<String> getErrorNames(Iterable<DataValue> errors) {
        Set<String> errorNames = new HashSet<String>();
        for (DataValue e : errors) {
            StructValue struct = toStruct(e, DATA_DEFINITION_NAME);
            errorNames.add(getDataDefinitionName(struct));
        }
        return errorNames;
    }

    private static StructValue toStruct(DataValue value,
            String expectedStructName) {
        if (!(value instanceof StructValue)) {
            throw new IntrospectionException(getMessage(
                    "vapi.introspection.data.struct.type.error",
                    expectedStructName, getValueName(value)));
        }
        StructValue struct = (StructValue) value;
        if (!expectedStructName.equals(struct.getName())) {
            throw new IntrospectionException(getMessage(
                    "vapi.introspection.data.struct.name.error",
                    expectedStructName, struct.getName()));
        }
        return struct;
    }

    private static String getValueName(DataValue value) {
        if (value != null) {
            return value.getType().name();
        } else {
            return "null";
        }
    }
}
