/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.provider.introspection;

import static com.vmware.vapi.provider.introspection.IntrospectionConstants.OPERATION_INTROSPECTION_SERVICE_ID;
import static com.vmware.vapi.provider.introspection.IntrospectionConstants.PROVIDER_INTROSPECTION_SERVICE_ID;
import static com.vmware.vapi.provider.introspection.IntrospectionConstants.SERVICE_INTROSPECTION_SERVICE_ID;

import java.util.Collections;
import java.util.HashSet;
import java.util.Set;

import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.DecoratorApiProvider;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.InterfaceDefinition;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.core.MethodDefinition;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.core.ProviderDefinition;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ErrorDefinition;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.internal.util.async.FilterAsyncHandle;
import com.vmware.vapi.provider.ApiInterface;
import com.vmware.vapi.provider.introspection.ApiIntrospection;

/**
 * {@link ApiProvider} decorator which provides remote introspection API (i.e.
 * {@link ApiProvider}) on top of a local introspection API (i.e.
 * {@link ApiIntrospection}).
 */
public final class IntrospectionFilter extends DecoratorApiProvider {

    private final ApiInterface providerIntrospection;
    private final ApiInterface serviceIntrospection;
    private final ApiInterface operationIntrospection;

    public IntrospectionFilter(ApiProvider decoratedProvider,
            ApiIntrospection introspection, Set<ErrorDefinition> extraErrors) {
        super(decoratedProvider);
        Validate.notNull(introspection);
        Validate.notNull(extraErrors);
        Validate.noNullElements(extraErrors);
        ApiIntrospection augmented =
                new AugmentedIntrospection(introspection, extraErrors);
        this.providerIntrospection = new ProviderIntrospectionService(
                augmented);
        this.serviceIntrospection = new ServiceIntrospectionService(
                augmented);
        this.operationIntrospection = new OperationIntrospectionService(
                augmented);
    }

    @Override
    public void invoke(String serviceId, String operationId, DataValue input,
            ExecutionContext ctx, AsyncHandle<MethodResult> asyncHandle) {
        InterfaceIdentifier ifaceId = new InterfaceIdentifier(serviceId);
        ApiInterface introspectionService = getIntrospectionService(ifaceId);
        if (introspectionService != null) {
            MethodIdentifier methodId =
                    new MethodIdentifier(ifaceId, operationId);
            introspectionService.invoke(ctx, methodId, input, asyncHandle);
        } else {
            decoratedProvider.invoke(serviceId, operationId, input, ctx,
                    asyncHandle);
        }
    }

    private ApiInterface getIntrospectionService(InterfaceIdentifier ifaceId) {
        if (PROVIDER_INTROSPECTION_SERVICE_ID.equals(ifaceId)) {
            return providerIntrospection;
        }
        if (SERVICE_INTROSPECTION_SERVICE_ID.equals(ifaceId)) {
            return  serviceIntrospection;
        }
        if (OPERATION_INTROSPECTION_SERVICE_ID.equals(ifaceId)) {
            return operationIntrospection;
        }
        return null;
    }

    /**
     * Decorator of {@link ApiIntrospection} which makes the introspection
     * services visible through the introspection API. For example the
     * introspection services would be returned by
     * {@link #getInterfaceIdentifiers(ExecutionContext, AsyncHandle)}.
     */
    private class AugmentedIntrospection implements ApiIntrospection {

        private final ApiIntrospection introspection;
        private final Set<ErrorDefinition> extraErrors;

        public AugmentedIntrospection(ApiIntrospection introspection,
                Set<ErrorDefinition> extraErrors) {
            this.introspection = introspection;
            this.extraErrors = Collections.unmodifiableSet(
                    new HashSet<ErrorDefinition>(extraErrors));
        }

        @Override
        public void getDefinition(ExecutionContext ctx,
                AsyncHandle<ProviderDefinition> asyncHandle) {
            introspection.getDefinition(ctx, asyncHandle);
        }

        @Override
        public void getInterfaceIdentifiers(ExecutionContext ctx,
                AsyncHandle<Set<InterfaceIdentifier>> asyncHandle) {
            AsyncHandle<Set<InterfaceIdentifier>> cb;
            // filter to append introspection service identifiers
            cb = new FilterAsyncHandle<Set<InterfaceIdentifier>>(
                    asyncHandle) {
                @Override
                protected Set<InterfaceIdentifier> filterResult(
                        Set<InterfaceIdentifier> result) {
                    Set<InterfaceIdentifier> augmented =
                            new HashSet<InterfaceIdentifier>(result);
                    augmented.add(PROVIDER_INTROSPECTION_SERVICE_ID);
                    augmented.add(SERVICE_INTROSPECTION_SERVICE_ID);
                    augmented.add(OPERATION_INTROSPECTION_SERVICE_ID);
                    return augmented;
                }
            };
            introspection.getInterfaceIdentifiers(ctx, cb);
        }

        @Override
        public void getInterface(ExecutionContext ctx,
                InterfaceIdentifier ifaceId,
                AsyncHandle<InterfaceDefinition> asyncHandle) {
            ApiInterface introspectionService = getIntrospectionService(ifaceId);
            if (introspectionService != null) {
                asyncHandle.setResult(introspectionService.getDefinition());
            } else {
                introspection.getInterface(ctx, ifaceId, asyncHandle);
            }
        }

        @Override
        public void getMethod(ExecutionContext ctx, MethodIdentifier methodId,
                AsyncHandle<MethodDefinition> asyncHandle) {
            ApiInterface introspectionService =
                    getIntrospectionService(methodId.getInterfaceIdentifier());
            if (introspectionService != null) {
                asyncHandle.setResult(augmentErrors(
                        introspectionService.getMethodDefinition(methodId)));
                return;
            } else {
                AsyncHandle<MethodDefinition> cb;
                // filter to augment error definitions
                cb = new FilterAsyncHandle<MethodDefinition>(asyncHandle) {
                    @Override
                    protected MethodDefinition filterResult(
                            MethodDefinition result) {
                        if (result == null) {
                            // method not found
                            return null;
                        }
                        return augmentErrors(result);
                    }
                };
                introspection.getMethod(ctx, methodId, cb);
            }
        }

        private Set<ErrorDefinition> augmentErrors(Set<ErrorDefinition> errors) {
            Set<ErrorDefinition> augmented =
                    new HashSet<ErrorDefinition>(errors);
            augmented.addAll(extraErrors);
            return augmented;
        }

        private MethodDefinition augmentErrors(MethodDefinition def) {
            return new MethodDefinition(def.getIdentifier(),
                    def.getInputDefinition(), def.getOutputDefinition(),
                    augmentErrors(def.getErrorDefinitions()));
        }
    }

}
