/* **********************************************************
 * Copyright 2013-2014, 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.provider.introspection;

import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.PROVIDER_GET_METHOD_ID;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.PROVIDER_INFO_DEF;
import static com.vmware.vapi.provider.introspection.IntrospectionConstants.PROVIDER_INTROSPECTION_SERVICE_ID;

import java.util.Collections;

import com.vmware.vapi.bindings.server.InvocationContext;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.core.ProviderDefinition;
import com.vmware.vapi.data.DataDefinition;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.StructDefinition;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.provider.ApiMethodBasedApiInterface;
import com.vmware.vapi.provider.introspection.ApiIntrospection;
import com.vmware.vapi.provider.introspection.SyncApiIntrospection;
import com.vmware.vapi.std.BuiltInDataFactory;

/**
 * This service provides operations to retrieve information about a
 * vAPI Provider. A provider represents a vAPI endpoint that is exposing a
 * collection of vAPI services.
 */
public class ProviderIntrospectionService extends ApiMethodBasedApiInterface {

    public ProviderIntrospectionService(ApiIntrospection introspection) {
        super(PROVIDER_INTROSPECTION_SERVICE_ID);
        Validate.notNull(introspection);
        registerMethod(new GetApiMethod(introspection));
    }

    public ProviderIntrospectionService(SyncApiIntrospection introspection) {
        this(new SyncToAsyncApiIntrospectionAdapter(introspection));
    }

    private static class GetApiMethod extends ApiIntrospectionMethodBase {
        private final static DataDefinition INPUT_DEF;
        private final static DataDefinition OUTPUT_DEF;

        static {
            INPUT_DEF = new StructDefinition(
                    BuiltInDataFactory.OPERATION_INPUT_STRUCT_NAME,
                    Collections.<String, DataDefinition>emptyMap());
            OUTPUT_DEF = PROVIDER_INFO_DEF;
        }

        public GetApiMethod(ApiIntrospection introspection) {
            super(PROVIDER_GET_METHOD_ID, INPUT_DEF, OUTPUT_DEF, null,
                    introspection);
        }


        @Override
        public void invoke(InvocationContext invocationContext,
                           DataValue input,
                           AsyncHandle<MethodResult> asyncHandle) {
            ExecutionContext executionContext =
                    invocationContext.getExecutionContext();
            introspection.getDefinition(executionContext,
                new IntrospectionAsyncHandle<ProviderDefinition>(asyncHandle) {
                    @Override
                    protected DataValue convert(ProviderDefinition result) {
                        return IntrospectionDataFactory.toProviderInfo(result);
                    }
                });
        }
    }
}
