/* **********************************************************
 * Copyright 2013-2014, 2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.provider.introspection;

import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.NOT_FOUND_ERROR_INFO_DEF;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.SERVICE_GET_METHOD_ID;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.SERVICE_GET_METHOD_ID_PARAM;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.SERVICE_INFO_DEF;
import static com.vmware.vapi.internal.provider.introspection.IntrospectionDataFactory.SERVICE_LIST_METHOD_ID;
import static com.vmware.vapi.provider.introspection.IntrospectionConstants.SERVICE_INTROSPECTION_SERVICE_ID;

import java.util.Arrays;
import java.util.Collections;
import java.util.Set;

import com.vmware.vapi.Message;
import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.bindings.server.InvocationContext;
import com.vmware.vapi.core.AsyncHandle;
import com.vmware.vapi.core.ErrorValueException;
import com.vmware.vapi.core.ExecutionContext;
import com.vmware.vapi.core.InterfaceDefinition;
import com.vmware.vapi.core.InterfaceIdentifier;
import com.vmware.vapi.core.MethodResult;
import com.vmware.vapi.data.DataDefinition;
import com.vmware.vapi.data.DataValue;
import com.vmware.vapi.data.ListDefinition;
import com.vmware.vapi.data.StringDefinition;
import com.vmware.vapi.data.StructDefinition;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.provider.ApiMethodBasedApiInterface;
import com.vmware.vapi.provider.introspection.ApiIntrospection;
import com.vmware.vapi.provider.introspection.SyncApiIntrospection;
import com.vmware.vapi.std.BuiltInDataFactory;
import com.vmware.vapi.std.StandardDataFactory;

/**
 * This service provides operations to retrieve information about the
 * services exposed by a vAPI Provider.
 */
public class ServiceIntrospectionService extends ApiMethodBasedApiInterface {

    public ServiceIntrospectionService(ApiIntrospection introspection) {
        super(SERVICE_INTROSPECTION_SERVICE_ID);
        Validate.notNull(introspection);
        registerMethod(new GetApiMethod(introspection));
        registerMethod(new ListApiMethod(introspection));
    }

    public ServiceIntrospectionService(SyncApiIntrospection introspection) {
        this(new SyncToAsyncApiIntrospectionAdapter(introspection));
    }

    private static class GetApiMethod extends ApiIntrospectionMethodBase {
        private final static DataDefinition INPUT_DEF;
        private final static DataDefinition OUTPUT_DEF;

        static {
            INPUT_DEF = new StructDefinition(
                    BuiltInDataFactory.OPERATION_INPUT_STRUCT_NAME,
                    Collections.<String, DataDefinition>singletonMap(
                            SERVICE_GET_METHOD_ID_PARAM,
                            StringDefinition.getInstance()));
            OUTPUT_DEF = SERVICE_INFO_DEF;
        }

        public GetApiMethod(ApiIntrospection introspection) {
            super(SERVICE_GET_METHOD_ID,
                  INPUT_DEF,
                  OUTPUT_DEF,
                  Collections.singleton(NOT_FOUND_ERROR_INFO_DEF),
                  introspection);
        }

        @Override
        public void invoke(InvocationContext invocationContext,
                           DataValue input,
                           AsyncHandle<MethodResult> asyncHandle) {
            ExecutionContext executionContext =
                    invocationContext.getExecutionContext();

            StructValue inputStruct = (StructValue) input;
            final String ifaceId =
                    inputStruct.getString(SERVICE_GET_METHOD_ID_PARAM);

            introspection.getInterface(executionContext,
                    new InterfaceIdentifier(ifaceId),
                    new IntrospectionAsyncHandle<InterfaceDefinition>(asyncHandle) {
                @Override
                protected DataValue convert(InterfaceDefinition result) {
                    if (result != null) {
                        return IntrospectionDataFactory.toServiceInfo(
                                result);
                    } else {
                        // interface Id not available - report NotFound
                        Message errorMsg = MessageFactory.getMessage(
                                "vapi.introspection.service.not_found",
                                ifaceId);

                        throw new ErrorValueException(
                                StandardDataFactory.createErrorValueForMessages(
                                                NOT_FOUND_ERROR_INFO_DEF,
                                                Arrays.asList(errorMsg)));
                    }
                }
            });
        }
    }

    private static class ListApiMethod extends ApiIntrospectionMethodBase {
        private final static DataDefinition INPUT_DEF;
        private final static DataDefinition OUTPUT_DEF;

        static {
            INPUT_DEF = new StructDefinition(
                    BuiltInDataFactory.OPERATION_INPUT_STRUCT_NAME,
                    Collections.<String, DataDefinition>emptyMap());
            OUTPUT_DEF = new ListDefinition(StringDefinition.getInstance());
        }

        public ListApiMethod(ApiIntrospection introspection) {
            super(SERVICE_LIST_METHOD_ID, INPUT_DEF, OUTPUT_DEF, null,
                    introspection);
        }

        @Override
        public void invoke(InvocationContext invocationContext,
                           DataValue input,
                           AsyncHandle<MethodResult> asyncHandle) {
            ExecutionContext executionContext =
                    invocationContext.getExecutionContext();
            introspection.getInterfaceIdentifiers(executionContext,
                new IntrospectionAsyncHandle<Set<InterfaceIdentifier>>(asyncHandle) {
                    @Override
                    protected DataValue convert(Set<InterfaceIdentifier> result) {
                        return IntrospectionDataFactory.toServiceIdList(result);
                    }
                });
        }
    }
}
