/* **********************************************************
 * Copyright 2021 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.internal.util;

import java.io.IOException;
import java.io.Reader;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.fasterxml.jackson.core.JsonFactory;
import com.fasterxml.jackson.core.JsonParseException;
import com.fasterxml.jackson.core.JsonParser;
import com.fasterxml.jackson.core.SerializableString;
import com.fasterxml.jackson.core.io.SerializedString;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;

/**
 * Utility methods related to JSON manipulation
 */
public class JacksonUtil {

    private static final Logger LOGGER = LoggerFactory.getLogger(JacksonUtil.class);

    /**
     * A default instance of a {@link JsonFactory} intended to be shared across the vapi runtime
     */
    public static final JsonFactory JSON_FACTORY = new JsonFactory();

    /**
     * Method to read the value of the specified field of an object read from the specified
     * jsonReader into a {@link JsonNode}
     * <p>
     * If the json content provided by the jsonReader does not represent an object or the object
     * does not contain a field with the specified name, {@code null} will be returned.
     *
     * @param jsonReader Reader to read the json content
     * @param fieldName The name of the field
     * @return A JsonNode that represents the value of the specified field or {@code null} if no
     *         object element with this field name is found
     */
    public static JsonNode readObjectFieldValue(Reader jsonReader, String fieldName)
    throws JsonParseException, IOException {
        Validate.notNull(fieldName);
        Validate.notNull(jsonReader);

        SerializableString serializedNodeName = new SerializedString(fieldName);
        ObjectMapper mapper = new ObjectMapper();

        try (JsonParser parser = JSON_FACTORY.createParser(jsonReader)) {
            // skip root structure
            parser.nextToken();

            while (!parser.nextFieldName(serializedNodeName)) {
                if (!parser.hasCurrentToken()) {
                    LOGGER.warn("JSON node {} not found", fieldName);
                    return null;
                }
                parser.skipChildren();
            }
            // move cursor to the value
            parser.nextToken();

            return mapper.readTree(parser);
        }
    }
}
