/* **********************************************************
 * Copyright (c) 2013-2022 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.util;

/**
 * Utility methods related to properties
 */
public class PropertyUtil {

    private PropertyUtil() {}

    /**
     * Retrieves a property value from the environment. First, checks if there's
     * a System property with the given {@code propertyName}. If there is, and
     * if its value is not blank, returns the value. Otherwise, checks if
     * there's an environment variable with the given {@code propertyName}.
     * If there is, and if its value is not blank, returns the value. Otherwise,
     * returns the given {@code defaultValue}.
     *
     * @param propertyName
     *        the name of the property whose value must be returned
     *
     * @param defaultValue
     *        the value, which should be returned, if there's no property
     *        with the given {@code propertyName}, or if the property's
     *        value is blank
     *
     * @return if there's a property with that name and its value isn't
     *         blank, returns that value; otherwise, returns the given
     *         {@code defaultValue}
     */
    public static String getProperty(String propertyName, String defaultValue) {
        Validate.notEmpty(propertyName);

        String value = System.getProperty(propertyName);
        if (StringUtils.isNotBlank(value)) {
            return value;
        }

        value = System.getenv(propertyName);
        if (StringUtils.isNotBlank(value)) {
            return value;
        }

        return defaultValue;
    }

    /**
     * Retrieves the value of the property with the given {@code propertyName}
     * by invoking method {@link #getProperty(String, String)}. If such a
     * property is found and its value is not blank, parses the value to a
     * boolean (see {@link Boolean#parseBoolean(String)}) and returns that
     * boolean. Otherwise, returns the given {@code defaultValue}.
     *
     * @param propertyName
     *        the name of the property whose value must be returned
     *
     * @param defaultValue
     *        the value, which should be returned, if there's no property
     *        with the given {@code propertyName}, or if the property's
     *        value is blank
     *
     * @return if there's a property with that name and its value isn't
     *         blank, parses the value to a boolean and returns that boolean;
     *         otherwise, returns the given {@code defaultValue}
     */
    public static boolean getBooleanProperty(String propertyName, boolean defaultValue) {
        String sValue = getProperty(propertyName, null);
        if (sValue != null) {
            return Boolean.parseBoolean(sValue);
        }
        return defaultValue;
    }
}
