/* **********************************************************
 * Copyright (c) 2017, 2019-2020 VMware, Inc.  All rights reserved. VMware Confidential
 * **********************************************************/

package com.vmware.vapi.internal.util;

import static com.vmware.vapi.MessageFactory.getMessage;

import java.util.Calendar;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.TimeZone;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.bindings.convert.ConverterException;

/**
 * <p>
 * Convert DateTime type to {@code ava.util.Calendar}. The DateTime type
 * is represented by <code>StringValue</code> in the API runtime, so this
 * converter converts between date-time string and {@code ava.util.Calendar}.
 * </p>
 * <p>
 * DateTime is a primitive string type that follows a date-time format defined
 * in RFC3339 which is a subset of ISO 8601.
 * DateTime string represents a complete date plus hours, minutes, seconds,
 * decimal fraction of a second and time-offset.
 *
 * <pre>
 *     YYYY-MM-DDThh:mm:ss<time-secfrac><time-offset>, e.g.:
 *     2017-12-30T11:40:50.123Z
 *     2017-12-30T11:40:50.123000000+02:00
 *     2017-12-30T11:40:50-05:00
 *
 *  where:
 *
 *   YYYY = four-digit year (years BC are not supported;
 *                           0001 = 1 AD is the first valid year,
 *                           0000 = 1 BC is not allowed)
 *   MM = two-digit month (01=January, ..., 12=December)
 *   DD = two-digit day of month (01 through 31)
 *   "T" = separator; appears literally in the string
 *   hh = two digits of hour (00 through 23; 24 NOT allowed; am/pm NOT allowed)
 *   mm = two digits of minute (00 through 59)
 *   ss = two digits of second (00 through 60)
 *        A leap second (ss=60) is accepted when all of the following rules apply
 *        for it when converted to UTC timezone:
 *          -year > 1971
 *          -leap second is added on last day of the month
 *          -leap second is added only to hour=23 and minute=59
 *
 *        Due to limitations of the {@code java.util.Calendar}, when a leap second
 *        is accepted, it is replaced with 59 and the milliseconds are set to 999, e.g.:
 *          -1972-06-30T23:59:60.000Z will be parsed to 1972-06-30T23:59:59.999Z
 *          -2016-12-31T23:59:60.123Z will be parsed to 2016-12-31T23:59:59.999Z
 *
 *   <time-secfrac> = optional decimal fraction of a second in format "." 1*DIGIT.
 *                  At least one digit is required after the decimal separator.
 *                  RFC3339 does not limit the number of digits. In current
 *                  implementation only the first 3 are significant, the rest
 *                  are ignored. To prevent attacks with long input when
 *                  fraction digits extend beyond 128 an error is returned.
 *
 *   <time-offset> = "Z" or <time-numoffset>
 *                   "Z" = UTC time zone designator; appears literally in the string
 *                   <time-numoffset> = "+" or "-" hh:mm
 * </pre>
 *
 * </p>
 * <p>
 * The {@link #toStringValue} method of this converter always converts
 * the received {@code java.util.Calendar} object to Gregorian time
 * (requirement of ISO8601). If the received {@code java.util.Calendar} object
 * does not use the Gregorian algorithm on the entire time-line, there might be
 * differences between the fields (year, month, day, hour, etc.) of the
 * {@code java.util.Calendar} object and the fields in its string representation.
 * For example "1801-03-04T05:06:07.000Z"in Gregorian calendar and
 * "1801-03-16T05:06:07.000Z" in Julian calendar both represent the same moment
 * of time (difference of 12 days). Furthermore consider that
 * {@code java.util.GregorianCalendar} by default uses Julian algorithm for
 * dates before 15 October 1582 and Gregorian algorithm for dates after that.
 * </p>
 */
public class Rfc3339DateTimeConverter {
    private static final char TIME_OFFSET_Z = 'Z';
    /** UTC timezone  */
    private static final TimeZone UTC_ZONE = TimeZone.getTimeZone("GMT+00:00");

    private static final int POWERS_OF_TEN[] = {1, 10, 100};

    /**
     * If this value is passed to
     * {@link java.util.GregorianCalendar#setGregorianChange(Date)}, it will
     * enforce the calendar object to use the Gregorian algorithm for the entire
     * time-line.
     */
    private static final Date PURE_GREGORIAN_CHANGE = new Date(Long.MIN_VALUE);

    /**
     * Regex pattern string which matches a DateTime string. There are groups
     * for each of the numeric fields (year, month, day, hour, minute, second,
     * decimal fraction, timezone sign, timezone, timezone hours, timezone minutes).
     * <pre>
     *     group(1) = year
     *     group(2) = month
     *     group(3) = day
     *     group(4) = hour
     *     group(5) = minute
     *     group(6) = second
     *     group(7) = decimal fraction (optional)
     *     group(8) = timezone sign
     *     group(9) = timezone hh:mm (optional)
     *
     * Samples:
     *     2017-12-30T11:40:50.123Z
     *     1: 2017
     *     2: 12
     *     3: 30
     *     4: 11
     *     5: 40
     *     6: 50
     *     7: .123
     *     8: Z
     *     9: null
     *
     *     2017-12-30T11:40:50-05:00
     *     1: 2017
     *     2: 12
     *     3: 30
     *     4: 11
     *     5: 40
     *     6: 50
     *     7: null
     *     8: -
     *     9: 05:00
     * </pre>
     */
    private static final String DATETIME_PATTERN_STRING =
            "(\\d{4})-(\\d{2})-(\\d{2})T(\\d{2}):(\\d{2}):(\\d{2})" +
            "(\\.\\d{1,128})?([Z,\\+,\\-])(\\d{2}:\\d{2})?";

    /**
     * Regex pattern object which matches a DateTime string.
     */
    private static final Pattern DATETIME_PATTERN = Pattern
            .compile(DATETIME_PATTERN_STRING);

    /**
     * Format string used to serialize a java.util.Calendar object to string.
     */
    private static final String DATETIME_FORMATTER_STRING =
            "%1$tY-%1$tm-%1$tdT%1$tH:%1$tM:%1$tS.%1$tLZ";

    /**
     * Parses a runtime datetime string representation to a {@link Calendar}
     *
     * @param datetime must not be <code>null</code>
     * @return {@link Calendar} representation of the datetime value.
     *         The calendar timezone will be UTC. must not be <code>null</code>
     */
    public GregorianCalendar fromStringValue(String datetime) {
        Matcher m = DATETIME_PATTERN.matcher(datetime);
        if (!m.matches()) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.datetime.deserialize.invalid.format",
                    datetime, DATETIME_PATTERN_STRING));
        }

        int year = Integer.parseInt(m.group(1));
        int month = Integer.parseInt(m.group(2));
        int day = Integer.parseInt(m.group(3));
        int hour = Integer.parseInt(m.group(4));
        int minute = Integer.parseInt(m.group(5));
        int second = Integer.parseInt(m.group(6));

        int millisecond = parseFraction(m.group(7));
        char tzSign = m.group(8).charAt(0);
        String tzHourMinute = m.group(9);

        // validate ranges
        if (year <= 0) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.datetime.deserialize.year.bc",
                    datetime));
        }
        if (month < 1 || month > 12) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.datetime.deserialize.month.invalid",
                    datetime));
        }
        if (day < 1 || day > 31) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.datetime.deserialize.day.invalid",
                    datetime));
        }
        if (hour > 23) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.datetime.deserialize.hour.invalid",
                    datetime));
        }
        if (minute > 59) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.datetime.deserialize.minute.invalid",
                    datetime));
        }
        // allow 60 as a leap second - further validation is applied below
        if (second > 60) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.datetime.deserialize.second.invalid",
                    datetime));
        }

        if (TIME_OFFSET_Z == tzSign) {
            if (tzHourMinute != null) {
                throw new ConverterException(getMessage(
                        "vapi.bindings.typeconverter.datetime.deserialize.timezone.invalid",
                        datetime));
            }
        } else {
            if (tzHourMinute == null) {
                throw new ConverterException(getMessage(
                        "vapi.bindings.typeconverter.datetime.deserialize.timezone.invalid",
                        datetime));
            }

            if ( Integer.parseInt(tzHourMinute.substring(0, 2)) > 23) {
                throw new ConverterException(getMessage(
                        "vapi.bindings.typeconverter.datetime.deserialize.timezone.hour.invalid",
                        datetime));
            }

            if (Integer.parseInt(tzHourMinute.substring(3)) > 59) {
                throw new ConverterException(getMessage(
                        "vapi.bindings.typeconverter.datetime.deserialize.timezone.minute.invalid",
                        datetime));
            }
        }

        TimeZone timeZone = buildTimeZone(tzSign, tzHourMinute);
        /*
         * Due to limitations of the {@code java.util.Calendar}, when a leap second
         * is accepted, it is replaced with 59 and the milliseconds are set to 999, e.g.:
         * -1972-06-30T23:59:60.000Z will be parsed to 1972-06-30T23:59:59.999Z
         * -2016-12-31T23:59:60.123Z will be parsed to 2016-12-31T23:59:59.999Z
         */
        if (second == 60) {
            if (isLeapSecondAllowed(year, month, day, hour, minute, timeZone)) {
                second = 59;
                millisecond = 999;
            } else {
                throw new ConverterException(getMessage(
                        "vapi.bindings.typeconverter.datetime.deserialize.second.leap.invalid",
                        datetime));
            }
        }

        try {
            GregorianCalendar calendar = new GregorianCalendar();
            calendar.clear();
            /*
             * Be strict and intolerant. This provides further validation
             * against violations such as 31st of February.
             */
            calendar.setLenient(false);
            calendar.setGregorianChange(PURE_GREGORIAN_CHANGE);
            calendar.setTimeZone(timeZone);
            calendar.set(Calendar.YEAR, year);
            // convert to zero-based month
            calendar.set(Calendar.MONTH, month - 1);
            calendar.set(Calendar.DAY_OF_MONTH, day);
            calendar.set(Calendar.HOUR_OF_DAY, hour);
            calendar.set(Calendar.MINUTE, minute);
            calendar.set(Calendar.SECOND, second);
            calendar.set(Calendar.MILLISECOND, millisecond);
            // force internal conversion and validation
            calendar.getTimeInMillis();
            return toUtcCalendar(calendar);
        } catch (RuntimeException ex) {
            throw new ConverterException(getMessage(
                    "vapi.bindings.typeconverter.datetime.deserialize.invalid.time",
                    datetime, ex.getMessage()), ex);
        }
    }

    /**
     * Verifies if the given date time can accommodate a leap second.
     * There is not any predictive algorithm for calculating it.
     * Based on the previously added 37 leap seconds
     * https://www.timeanddate.com/time/leap-seconds-future.html
     * the following rules are used and they all have to to apply for it
     * when converted to UTC timezone:
     * <p>
     * <pre>
     *    - year > 1971
     *    - leap second is added on last day of the month
     *    - leap second is added only to hour=23 and minute=59
     * </pre>
     * </p>
     * @param year year
     * @param month month (1 = January, 2 February, ...)
     * @param day day of the month
     * @param hour hour
     * @param minute minute
     * @param timeZone timezone
     * @return true if for the given moment of time is possible to hold a leap second
     *         false otherwise
     */
    private boolean isLeapSecondAllowed(int year, int month, int day,
                                        int hour, int minute, TimeZone timeZone) {
        if (year < 1972) {
            return false;
        }
        GregorianCalendar calendar = new GregorianCalendar();
        calendar.clear();
        calendar.setGregorianChange(PURE_GREGORIAN_CHANGE);
        calendar.setTimeZone(timeZone);
        calendar.set(Calendar.YEAR, year);
        calendar.set(Calendar.MONTH, month - 1);
        calendar.set(Calendar.DAY_OF_MONTH, day);
        calendar.set(Calendar.HOUR_OF_DAY, hour);
        calendar.set(Calendar.MINUTE, minute);
        // force internal conversion and validation
        calendar.getTimeInMillis();
        // the leap second rules apply to utc timezone
        Calendar utcCal = toUtcCalendar(calendar);
        int utcHour = utcCal.get(Calendar.HOUR_OF_DAY);
        int utcMinute = utcCal.get(Calendar.MINUTE);
        if (utcHour == 23 && utcMinute == 59 && isLastDayOfMonth(utcCal)) {
            return true;
        }
        return false;
    }

    private boolean isLastDayOfMonth(Calendar cal) {
        int month = cal.get(Calendar.MONTH);
        cal.add(Calendar.DAY_OF_MONTH, 1);
        return month != cal.get(Calendar.MONTH);
    }

    /**
     * Converts the given <code>java.util.Calendar</code> object to Gregorian time
     * string representation.
     * <p>
     * The default format is used: YYYY-MM-DDThh:mm:ss.SSSZ.
     * </p>
     * @param calendar a calendar
     * @return string representation of the calendar
     */
    public String toStringValue(Calendar calendar) {
        GregorianCalendar utcCalendar = toUtcCalendar(calendar);
        /*
         * Make sure we don't get tricked into serializing something which
         * violates the format.
         */
        if (isYearBC(utcCalendar)) {
            throw new CoreException(getMessage(
                    "vapi.bindings.typeconverter.datetime.serialize.year.bc",
                    calendar.toString()));
        }
        if (utcCalendar.get(Calendar.YEAR) > 9999) {
            throw new CoreException(getMessage(
                    "vapi.bindings.typeconverter.datetime.serialize.year.too.big",
                    calendar.toString()));
        }
        return String.format(DATETIME_FORMATTER_STRING, utcCalendar);
    }

    /**
     * Builds a time zone from time zone sign and time zone minutes and hours.
     *
     * @param tzSign 'Z', '+' and '-'
     * @param tzHourMinute a string in format "hh:mm". It can be null for tzSign is 'Z'.
     * @return TimeZone
     */
    private TimeZone buildTimeZone(char tzSign, String tzHourMinute) {
        if (TIME_OFFSET_Z == tzSign) {
                return UTC_ZONE;
        }
        StringBuilder sb = new StringBuilder(9);
        sb.append("GMT").append(tzSign).append(tzHourMinute);
        return TimeZone.getTimeZone(sb.toString());
    }

    /**
     * Parses decimal fraction from string and convert it to milliseconds.
     * At most 3 most significant digits are used for calculation of milliseconds.
     * The rest are ignored.
     *
     * @param strFraction decimal fraction of a second that starts with ".".
     * It can be null.
     * @return seconds fractional part in milliseconds
     */
    private int parseFraction(String strFraction) {
        if (strFraction == null) {
            return 0;
        }
        int endIndex = strFraction.length() < 4 ? strFraction.length() : 4;
        int significand = Integer.parseInt(strFraction.substring(1, endIndex));

        // shift so that .01 and .1 are 10 and 100 millisecond respectively
        return significand * POWERS_OF_TEN[4 - endIndex];
    }

    /**
     * Checks whether a calendar has a year from the BC era.
     *
     * @param c calendar
     * @return whether the calendar object is from the BC era
     */
    private static boolean isYearBC(GregorianCalendar c) {
        return c.get(Calendar.ERA) == GregorianCalendar.BC;
    }

    /**
     * Converts the specified calendar to an equivalent calendar in the UTC
     * timezone.
     *
     * @param calendar  calendar
     * @return calendar in UTC timezone
     */
    private static GregorianCalendar toUtcCalendar(Calendar calendar) {
        GregorianCalendar utcCalendar = new GregorianCalendar();
        utcCalendar.clear();
        utcCalendar.setLenient(false);
        utcCalendar.setGregorianChange(PURE_GREGORIAN_CHANGE);
        utcCalendar.setTimeZone(UTC_ZONE);
        utcCalendar.setTimeInMillis(calendar.getTimeInMillis());
        return utcCalendar;
    }

}
