/* **********************************************************
 * Copyright (c) 2018-2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.client.http;

import java.util.Arrays;
import java.util.List;

import com.vmware.vapi.core.HttpResponse;
import com.vmware.vapi.core.HttpResponseAccessor;
import com.vmware.vapi.internal.util.Validate;

/**
 * Composite {@link HttpResponseAccessor}.
 *
 * <p>Composes several instances {@code HttpResponseAccessor} instances
 * and provides access to the HTTP response for each of them.
 *
 * <p>Per {@link HttpResponseAccessor#access(HttpResponse)} none of the
 * instances is supposed to throw a {@code RuntimeException}. In case some
 * of them do, the rest of the composed instances will not be provided with
 * the HTTP response.
 */
public class CompositeResponseAccessor implements HttpResponseAccessor {
    private final List<HttpResponseAccessor> accessors;

    public CompositeResponseAccessor(List<HttpResponseAccessor> accessors) {
        Validate.noNullElements(accessors);
        this.accessors = accessors;
    }

    public CompositeResponseAccessor(HttpResponseAccessor... accessors) {
        Validate.noNullElements(accessors);
        this.accessors = Arrays.asList(accessors);
    }

    @Override
    public void access(HttpResponse response) {
        for (HttpResponseAccessor acc : accessors) {
            acc.access(response);
        }
    }
}
