/* **********************************************************
 * Copyright 2011-2014 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.protocol.server.rpc.http.impl;

import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import javax.net.ssl.SSLSocketFactory;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.internal.protocol.server.rpc.http.impl.BasicLifecycleManager;
import com.vmware.vapi.protocol.server.rpc.http.Endpoint;
import com.vmware.vapi.protocol.server.rpc.http.Filter;
import com.vmware.vapi.protocol.server.rpc.http.LifecycleListener;
import com.vmware.vapi.protocol.server.rpc.http.LifecycleManager;
import com.vmware.vapi.protocol.server.rpc.http.Server;
import com.vmware.vapi.protocol.server.rpc.http.ServerConfigurator;
import com.vmware.vapi.protocol.server.rpc.http.Service;
import com.vmware.vapi.protocol.server.rpc.http.StaticContentService;

/**
 * Base class implementing Server interface with common code.
 */
public abstract class AbstractServer implements Server, LifecycleManager {

   /** Logger */
   protected static final Logger _logger =
           LoggerFactory.getLogger(AbstractServer.class);

   protected static final Service[] NO_SERVICES = new Service[0];
   protected static final Filter[] NO_FILTERS = new Filter[0];

   public static final int DEFAULT_PORT = 8080;

   /** Server listening endpoints */
   protected Endpoint[] _endpoints;

   /** Services to handle HTTP requests */
   protected Service[] _services;

   /** Filters for HTTP requests and responses */
   protected Filter[] _filters;

   /** Common lifecycle for all server components */
   protected final LifecycleManager _lifecycleManager;

   protected ServerConfigurator _serverConfig;

   /** List of enabled cipher suites */
   private static final String[] ENABLED_SSL_CIPHERS = new String[] {
      "TLS_ECDHE_ECDSA_WITH_AES_256_CBC_SHA",
      "TLS_ECDHE_RSA_WITH_AES_256_CBC_SHA",
      "TLS_ECDH_ECDSA_WITH_AES_256_CBC_SHA",
      "TLS_ECDH_RSA_WITH_AES_256_CBC_SHA",
      "TLS_DHE_RSA_WITH_AES_256_CBC_SHA",
      "TLS_DHE_DSS_WITH_AES_256_CBC_SHA",
      "TLS_DH_RSA_WITH_AES_256_CBC_SHA",
      "TLS_DH_DSS_WITH_AES_256_CBC_SHA",
      "TLS_RSA_WITH_AES_256_CBC_SHA",
      "TLS_DHE_RSA_WITH_AES_128_CBC_SHA",
      "TLS_RSA_WITH_AES_128_CBC_SHA",
      "TLS_DHE_DSS_WITH_AES_128_CBC_SHA",
      "TLS_DH_RSA_WITH_AES_128_CBC_SHA",
      "TLS_DH_DSS_WITH_AES_128_CBC_SHA"
   };

   private String[] _enabledCiphers = ENABLED_SSL_CIPHERS;

   /**
    * Prepares default servlet for serving static content. Each Server
    * implementation receives a StaticContentService and must return a service
    * configured to serve static content.
    *
    * @param staticSrv a static content service definition
    * @return a static content service prepared to be used with server
    *         implementation
    */
   protected abstract Service prepareDefaultServlet(StaticContentService staticSrv);

   public AbstractServer() {
      _endpoints = new Endpoint[] { new HttpEndpoint(DEFAULT_PORT) };
      _services = NO_SERVICES;
      _filters = NO_FILTERS;
      _lifecycleManager = new BasicLifecycleManager();
   }

   @Override
   public void setEndpoints(Endpoint[] endpoints) {
      validateArray(endpoints);
      _endpoints = endpoints;
   }

   @Override
   public void setServices(Service[] services) {
      validateArray(services);
      _services = services;
      prepareStaticContentServices();
   }

   @Override
   public void setFilters(Filter[] filters) {
      validateArray(filters);
      _filters = filters;
   }

   @Override
   public void setServerConfigurator(ServerConfigurator serverConfig) {
      _serverConfig = serverConfig;
   }

   protected static void validateArray(Object[] array) {
      if (array == null || array.length == 0) {
         throw new IllegalArgumentException("Empty array");
      }

      for (int i = 0; i < array.length; i++) {
         if (array[i] == null) {
            throw new IllegalArgumentException("Empty array element at index " + i);
         }
      }
   }

   protected void prepareStaticContentServices() {
      for (int i = 0; i < _services.length; i++) {
         if (_services[i] instanceof StaticContentServiceImpl) {
            StaticContentService staticSrv = (StaticContentService) _services[i];
            if (staticSrv.getContentBase() == null) {
               throw new IllegalArgumentException(
                     "contentBase must not be null for service " + staticSrv);
            }
            _services[i] = prepareDefaultServlet(staticSrv);
         }
      }
   }

   protected void applyServerConfigurator () {
      if (_serverConfig != null) {
         if (_logger.isDebugEnabled()) {
            _logger.debug("Applying server configurator " + _serverConfig +
                  " to server " + this);
         }
         _serverConfig.configure(this);
      }
   }

   @Override
   public void addLifecycleListener(LifecycleListener lifecycleListener) {
      _lifecycleManager.addLifecycleListener(lifecycleListener);
   }

   @Override
   public void setLifecycleListeners(List<LifecycleListener> lifecycleListeners) {
      _lifecycleManager.setLifecycleListeners(lifecycleListeners);
   }

   @Override
   public void shutdown() {
      _lifecycleManager.shutdown();
   }

   @Override
   public void shutdownNow() {
      _lifecycleManager.shutdownNow();
   }

   /**
    * Configures the enabled SSL ciphers.
    *
    * @param enabledCiphers the enabled SSL ciphers.
    * @throws NullPointerException if enabledCiphers is null or empty.
    */
   public void setEnabledSSLCiphers(String[] enabledCiphers) {
      if (enabledCiphers == null || enabledCiphers.length == 0) {
         throw new NullPointerException("Can not specify empty list");
      }
      _enabledCiphers = enabledCiphers;
   }

   /**
    * Returns an array with supported disabled cipher suites.
    *
    * @return array of disabled cipher suites that are supported on the JVM
    */
   protected String[] getDisabledSSLCiphers() {
      // Get enabled cipher suites
      SSLSocketFactory sslf = (SSLSocketFactory)SSLSocketFactory.getDefault();
      Set<String> ciphers = new HashSet<String>(
            Arrays.asList(sslf.getDefaultCipherSuites()));

      // Remove cipher suites in _enabledCiphers array
      ciphers.removeAll(Arrays.asList(_enabledCiphers));

      return ciphers.toArray(new String[ciphers.size()]);
   }

   /**
    * Returns an array with supported enabled cipher suites.
    *
    * @return array of enabled cipher suites that are supported on the JVM
    */
   protected String[] getEnabledSSLCiphers() {
      // Get enabled cipher suites
      SSLSocketFactory sslf = (SSLSocketFactory)SSLSocketFactory.getDefault();
      Set<String> ciphers = new HashSet<String>(
            Arrays.asList(sslf.getDefaultCipherSuites()));

      // Filter ciphers based on _enabledCiphers array
      ciphers.retainAll(Arrays.asList(_enabledCiphers));

      return ciphers.toArray(new String[ciphers.size()]);
   }
}