/* **********************************************************
 * Copyright 2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.protocol.server.rpc.http.impl;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.common.http.HttpConstants;
import com.vmware.vapi.protocol.server.rpc.RequestReceiver;
import com.vmware.vapi.protocol.server.rpc.http.MediaTypeResolver;

/**
 * Resolver which is based on a map of media types to content handlers.
 */
public final class MediaTypeResolverImpl implements MediaTypeResolver {

    private final Map<String, RequestReceiver> mediaTypeToHandler;

    /**
     * Creates a resolver which processes only JSON content.
     *
     * @param jsonHandler content handler; must not be <code>null</code>
     * @return resolver
     */
    public static MediaTypeResolver newJsonResolver(RequestReceiver jsonHandler) {
        Validate.notNull(jsonHandler);
        Map<String, RequestReceiver> m = new HashMap<String, RequestReceiver>();
        m.put(HttpConstants.CONTENT_TYPE_JSON, jsonHandler);
        return new MediaTypeResolverImpl(m);
    }

    /**
     * Constructor which accepts a mapping of media type to content handler.
     * Wildcards are not supported in media types.
     *
     * @param mediaTypeToHandler maps a media/MIME type to a content handler
     */
    public MediaTypeResolverImpl(
            Map<String, RequestReceiver> mediaTypeToHandler) {
        Validate.notNull(mediaTypeToHandler);
        for (Map.Entry<String, RequestReceiver> e :
            mediaTypeToHandler.entrySet()) {
            Validate.notEmpty(e.getKey());
            Validate.notNull(e.getValue());
        }
        this.mediaTypeToHandler =
                Collections.unmodifiableMap(mediaTypeToHandler);
    }

    @Override
    public RequestReceiver getHandler(String mediaType) {
        Validate.notNull(mediaType);
        return mediaTypeToHandler.get(mediaType);
    }



}
