/* **********************************************************
 * Copyright (c) 2011-2015, 2019 VMware, Inc. All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.server;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.config.Configurator;
import com.vmware.vapi.config.PropertyConfigurator;
import com.vmware.vapi.config.SpringConfigurator;
import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.protocol.ProtocolConnection;
import com.vmware.vapi.protocol.ProtocolHandler;
import com.vmware.vapi.protocol.local.LocalConnection;
import com.vmware.vapi.protocol.local.LocalProtocol;

/**
 * Starts an ApiProvider server.
 */
public class Server implements ServerInterface {
    static Logger logger = LoggerFactory.getLogger(Server.class);

    private volatile Thread mainLoop;
    private ApiProvider primary;
    private List<ProtocolHandler> handlers;
    private LocalProtocol localProtocol;
    private volatile boolean shutdown;
    /** Mechanism to wait for shutdown completion. */
    private volatile CountDownLatch shutdownLatch = new CountDownLatch(1);

    /**
     * Creates an <code>{@link ApiProvider}</code> instance configured
     * according to the provided properties object.
     *
     * @param props       properties object
     *
     * @throws Exception
     */
    public Server(Properties props) throws Exception {
        this(new PropertyConfigurator(props));
    }

    /**
     * Creates an <code>{@link ApiProvider}</code> instance configured
     * by the provided <code>Configurator</code>.
     *
     * @param configurator  configurator instance
     */
    public Server(Configurator configurator) {
        shutdown = false;
        primary = configurator.getApiProvider();
        handlers = configurator.getProtocolHandlers();
        localProtocol = new LocalProtocol(primary);
        for (ProtocolHandler protocol : handlers) {
            protocol.start();
        }
    }

    /** Stops the server. */
    public void stop() {
        shutdown = true;
        // Force the main loop to exit so we clean up our resources
        if (mainLoop != null) {
            mainLoop.interrupt();
        }
    }

    /**
     * Wait for the server shutdown to complete. If the server is already
     * shutdown, the method will return <code>true</code> immediately.
     *
     * @param timeoutMillis     maximum time to wait for the shutdown to
     *                          complete (in milliseconds)
     * @return                  whether the shutdown completed or the timeout
     *                          expired
     * @throws InterruptedException the wait has been interrupted
     */
    public boolean waitForShutdown(long timeoutMillis)
            throws InterruptedException {
        return shutdownLatch.await(timeoutMillis, TimeUnit.MILLISECONDS);
    }

    /** Runs the server. */
    public void run() {
        // TODO: this Server thing only works (can be shutwodn) if invoked
        //       in a special/dedicated thread (which is just burned out, for
        //       all the time the server is up)
        //       why should we require this?
        mainLoop = Thread.currentThread();
        while (!shutdown) {
            try {
                Thread.sleep(1000);
            } catch (InterruptedException e) {
                // Exception can be thrown when stop() is called
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        for (ProtocolHandler handler: handlers) {
            handler.stop();
        }
        // shutdown is complete
        shutdownLatch.countDown();
    }

    /**
     * Returns local connection to the primary API provider. This connection can
     * be used only by clients inside the same process.
     *
     * @return connection which involves no networking
     */
    public ProtocolConnection getLocalConnection() {
        return new LocalConnection(localProtocol);
    }

    /**
     * Retrieves a resource from the classpath.
     *
     * @param classpathResource path of a resource on the classpath
     * @return resource stream or <code>null</code> if the resource does not
     *         exist
     */
    private static InputStream getClasspathResource(String classpathResource) {
        if (!classpathResource.startsWith("/")) {
            classpathResource = "/" + classpathResource;
        }
        return Server.class.getResourceAsStream(classpathResource);
    }

    private static Configurator buildSpringConfigurator(String configFileName) {
        return new SpringConfigurator(Arrays.asList(configFileName));
    }

    private static Configurator buildPropertiesConfigurator(String propFileName)
            throws IOException, FileNotFoundException {
        Properties props = new Properties();
        File propFile = new File(propFileName);

        InputStream inPropsStream = null;
        try {
            inPropsStream = propFile.isFile() ? new FileInputStream(propFile)
                                              : getClasspathResource(propFileName);
            if (inPropsStream == null) {
                throw new IllegalArgumentException("Property file '"
                        + propFileName + "' does not exist on the file system"
                        + " nor on the classpath");
            }

            props.load(inPropsStream);
        } finally {
            if (inPropsStream != null) {
                inPropsStream.close();
            }
        }

        return new PropertyConfigurator(props);
    }
    /**
     * Program entry point; runs the server using the configuration
     * specified in the properties file provided as a command-line
     * argument.
     *
     * @param args  program arguments; should contain a single entry
     *              specifying the path to the properties file or Spring
     *              XML config file for the server instance.
     * @throws Exception problem with the starting the server
     */
    public static void main(String[] args) throws Exception {
        if (args.length != 1) {
            System.err.println("Please specify provider configuration as argument (either "
                    + ".properties or Spring .xml configuration");
            System.exit(1);
        }

        String configFileName = args[0];
        Configurator configurator = null;
        if (configFileName.endsWith(".xml")) {
            configurator = buildSpringConfigurator(configFileName);
        } else if (configFileName.endsWith(".properties")) {
            configurator = buildPropertiesConfigurator(configFileName);
        } else {
            System.err.println("Unsupported provider configuration (expected .properties "
                    + "or Spring .xml config)");
            System.exit(1);
        }

        Server s = new Server(configurator);
        s.run();
    }
}
