/* **********************************************************
 * Copyright 2013-2014, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings;

import java.io.Serializable;
import java.util.HashMap;
import java.util.Map;

import com.vmware.vapi.internal.util.Validate;

/**
 * Base class for all Java mappings of interface definition language
 * enumerated types.
 *
 * <p>The interface language definition type system includes enumerated types.
 * For every such enumerated type a class which extends this class is
 * generated (referred to as <i>enumeration class</i>). The generated class contains
 * public static member fields initialized with instances of the class for
 * each enumeration constant.
 *
 * <p>In addition, the generated <i>enumeration class</i> contains inner native Java
 * {@code enum} declaraing the same constants. This native enum can be used for more
 * convenient handling of enumerations, e.g. in usage in {@code switch} statements.
 * The simple name of the inner native enum is always {@code Values}.
 *
 * <p> Example of generated "enumeration class" and its inner native enum:
 * <pre>
 *  public final class SourceType extends com.vmware.vapi.bindings.ApiEnumeration<SourceType> {
 *
 *      public static final SourceType FILE = new SourceType("FILE");
 *
 *      public static final SourceType REMOTE = new SourceType("REMOTE");
 *
 *      public enum Values {
 *          FILE,
 *          REMOTE,
 *          _UNKNOWN,
 *      }
 *      ...
 *  }
 * </pre>
 *
 * <p>Example of obtaining native enum constant corresponding to given
 * </i>enumeration class</i> constant
 * <pre>
 *   SourceType sourceType = SourceType.FILE;
 *   SourceType.Values nativeEnumValue = sourceType.getEnumValue();
 *
 *   // can use it in switch statement
 *   switch (nativeEnumValue) {
 *      case: ...
 *   }
 * </pre>
 *
 * <p><b>Note:</b>The generated </i>enumeration class</i> defines constants
 * representing the values in the <b>current version</b> of the enumerated type.
 * Newer versions of the enumerated type may contain additional values. Such
 * newer values are not known at generation and compile time. They are represented
 * by the {@code _UNKNOWN} constant of the native enum and by an instance of the
 * </i>enumeration class</i> which is not referred by any of the static member
 * fields of the class. For the </i>enumeration class</i> instances conversion
 * to and from {@code String} is supported, which allows usage of such newer
 * constants.
 *
 * <pre>
 *  // create enumeration class instance for the newer value FOLDER
 *  SourceType sourceType = SourceType.valueOf("FOLDER");
 *
 *  // obtain String representation of the newer value (e.g. received from a
 *  // server which supports newer version of the API
 *  SourceType sourceType = ...
 *  String strValue = sourceType.name();
 * </pre>
 */
public abstract class ApiEnumeration<E extends ApiEnumeration<E>>
        implements Serializable {

    private static final long serialVersionUID = 1L;

    private final String name;

    /**
     * @param name enumeration value; must not be <code>null</code>
     */
    protected ApiEnumeration(String name) {
        Validate.notNull(name);
        this.name = name;
    }

    /**
     * @param values an array of enumeration values; must not be
     *               <code>null</code>
     * @return map of the values, where the keys are the names of the values
     */
    protected static <T extends ApiEnumeration<T>> Map<String, T> buildNameMap(
            T[] values) {
        Validate.notNull(values);
        Map<String, T> byName = new HashMap<String, T>();
        for (T value : values) {
            byName.put(value.name(), value);
        }
        return byName;
    }

    /**
     * @return the name of the enumeration value
     */
    public String name() {
        return name;
    }

    @Override
    public boolean equals(Object obj) {
        if (this == obj) {
            return true;
        }
        if (!getClass().isInstance(obj)) {
            return false;
        }
        ApiEnumeration<?> other = (ApiEnumeration<?>) obj;
        return name().equals(other.name());
    }

    @Override
    public int hashCode() {
        return name.hashCode();
    }

    @Override
    public String toString() {
        return name();
    }

}
