/* **********************************************************
 * Copyright (c) 2013, 2018, 2021 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings;

import java.util.Set;

import com.vmware.vapi.CoreException;
import com.vmware.vapi.bindings.type.StructType;
import com.vmware.vapi.client.exception.BindingsException;
import com.vmware.vapi.data.DataValue;

/**
 * Interface implemented by all generated static bindings structures.
 *
 * <p>A static structure provides getters/setters for access to the fields known
 * at generation time. It may contain additional <i>dynamic fields</i> which can be
 * accessed using method from this interface.
 */
public interface StaticStructure extends Structure {

    /**
     * <b>WARNING:</b> Internal method, subject to change in future versions.
     *
     * Returns {@link StructType} instance representing the static bindings
     * type for this <code>Structure</code>.
     *
     * @return the static bindings <code>StructType</code>
     */
    StructType _getType();

    /**
     * Validates the state of the discriminated unions and HasFieldsOf
     * restrictions (if any) in this Java language binding <code>Structure</code>.
     *
     * <p>More precisely, for each union: given value of the discriminant/tag
     * field (<code>@UnionTag</code> in the IDL) of an union in this structure,
     * validates that:
     *  <li>all fields of the union required for that value of the
     *      discriminant are set
     *  <li>fields of the union that are not allowed for that value of the
     *      discriminant are not set
     *  <li>any other fields of the structure (not participating in the union)
     *      do not affect the validation result
     *
     * @throws CoreException if some constraint is not satisfied
     */
    void _validate();

    /**
     * Returns a string representation of this structure. The result is pretty
     * printed in order to be easy to read. It is intended to be used
     * for different debugging/tracing purposes, e.g. dumping structures
     * into logging traces.
     *
     * <p>In general no code should depend on the format of the resulting
     * string. E.g. there must be no attempts to parse for re-creation of the
     * original instance.
     *
     * @return a string representation of this static structure instance
     */
    @Override
    String toString();

    /**
     * Sets a dynamic field value. The dynamic filed is extra filed that is not
     * part of the static structure definition, i.e. not represented with
     * getter/setter.
     *
     * @param fieldName canonical name of the dynamic field to set
     * @param fieldValue the value of the field
     * @throws BindingsException in case there is a field in the static
     *                           structure definition with that name
     */
    public void _setDynamicField(String fieldName, DataValue fieldValue);

    /**
     * Get a dynamic field value, doesn't allow access to static fields (which
     * have getters/setters).
     *
     * @param fieldName the canonical name of the dynamic field
     * @return the {@link DataValue} of the dynamic field
     * @throws BindingsException if there is no dynamic field with that name
     */
    public DataValue _getDynamicField(String fieldName);

    /**
     * Get the names of the dynamic fields in the structure.
     *
     * @return set of the canonical field names for the dynamic fields
     */
    public Set<String> _getDynamicFieldNames();
}
