/* **********************************************************
 * Copyright 2012 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings.client;

import java.util.concurrent.locks.Condition;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import com.vmware.vapi.std.Progress;

/**
 * This utility provides a synchronous view of an asynchronous callback.
 */
public class AsyncCallbackSyncAdapter<T> extends AsyncCallback<T> {

    private final Lock lock = new ReentrantLock();
    private final Condition set = lock.newCondition();
    private boolean completed = false;

    private T result;
    private RuntimeException error;

    /**
     * Returns the result or throws the error of this callback.
     *
     * @return the result, if set by {@link #setResult(Object)}
     * @throws InterruptedException waiting for the result is interrupted
     * @throws RuntimeException  the error, if set by
     *                           {@link #setError(RuntimeException)}
     */
    public T get() throws InterruptedException {
        lock.lock();
        try {
            while (!completed) {
                set.await();
             }
             if (error == null) {
                return result;
             } else {
                 throw error;
             }
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void onProgress(Progress progress) {
    }

    @Override
    public void onResult(T result) {
        lock.lock();
        try {
            this.result = result;
            this.completed = true;
            set.signalAll();
        } finally {
            lock.unlock();
        }
    }

    @Override
    public void onError(RuntimeException error) {
        lock.lock();
        try {
            this.error = error;
            this.completed = true;
            set.signalAll();
        } finally {
            lock.unlock();
        }
    }

}
