/* **********************************************************************
 * Copyright 2012-2013, 2019 VMware, Inc. All rights reserved. VMware Confidential
 * ********************************************************************
 */
package com.vmware.vapi.bindings.type;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;

import com.vmware.vapi.internal.util.Validate;

/**
 * Description of an ID type.
 */
public class IdType implements Type {
    private String resourceType;
    private List<String> resourceTypes;
    private String resourceTypeFieldName;

    /**
     * Constructor.
     *
     * <p>Creates an instance for given resource type, i.e. the value
     * of <code>@Resource(...)</code> from the IDL
     *
     * @param resourceType resource type of the ID;
     *        must not be <code>null</code>
     * @throws IllegalArgumentException if the resource type is
     *         <code>null</code>
     */
    public IdType(String resourceType) {
        Validate.notNull(resourceType, "resourceType must not be null");
        this.resourceType = resourceType;
    }

    /**
     * Constructor for the polymorphic resource ID case.
     *
     * <p>Creates an instance for given allowed resource types and resource
     * type field. This are the values specified at the
     * <code>@Resource(String[] allowedTypes, typeFieldName)</code> in the IDL
     *
     * @param resourceTypes the allowed resource types
     * @param resourceTypeFieldName the name of the field specifying the
     *        resource type
     * @throws IllegalArgumentException if both arguments are <code>null</code>
     */
    public IdType(String[] resourceTypes, String resourceTypeFieldName) {
        Validate.isTrue(
                (resourceTypes != null && resourceTypes.length != 0)
                    || resourceTypeFieldName != null,
                "IdType requires either resource types or resource " +
                    "type holder field specified.");

        if (resourceTypes != null) {
            this.resourceTypes =
                    Collections.unmodifiableList(Arrays.asList(resourceTypes));
        }
        this.resourceTypeFieldName = resourceTypeFieldName;
    }

    @Override
    public void accept(TypeVisitor visitor) {
        visitor.visit(this);
    }

    /**
     * Returns the resource type for the <code>ID</code>.
     *
     * @return <code>String</code> representing the resource type
     */
    public String getResourceType() {
        return resourceType;
    }

    /**
     * Returns the allowed resource types for polymorphic resource
     * <code>ID</code>.
     *
     * @return <code>List&lt;String&gt;</code> representing the resource type
     */
    public List<String> getResourceTypes() {
        return resourceTypes;
    }

    /**
     * Returns the field specifying the resource type for polymorphic
     * resource <code>ID</code> in canonical form.
     *
     * @return <code>String</code> representing the resource type
     */
    public String getResourceTypeFieldName() {
        return resourceTypeFieldName;
    }
}