/* **********************************************************
 * Copyright (c) 2012-2015, 2018-2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/
package com.vmware.vapi.bindings.type;

import java.lang.reflect.Method;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.NoSuchElementException;
import java.util.Set;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.vmware.vapi.bindings.convert.ConverterException;
import com.vmware.vapi.core.MethodIdentifier;
import com.vmware.vapi.data.ConstraintValidationException;
import com.vmware.vapi.data.StructValue;
import com.vmware.vapi.internal.data.ConstraintValidator;
import com.vmware.vapi.internal.util.Validate;

/**
 * Description of a structure type.
 */
public class StructType implements Type {
    private static final Logger logger = LoggerFactory.getLogger(StructType.class);
    private final String name;
    private final String discriminatedBy;
    private final String discriminator;
    private final Map<String, Type> fields;
    private final Class<?> bindingClass;
    private final List<ConstraintValidator> validators;
    private final boolean isModel;
    private final List<String> modelKeyFields;
    private final Map<String, FieldInfo> fieldInfos;
    private final Iterable<String> fieldWithSetterNames;

    /**
     * Field name details structure.
     */
    public static class FieldNameDetails {
        private final String canonicalName;
        private final String mixedCaseName;
        private final String getterName;
        private final String setterName;

        /**
         * Create a field name details instance.
         *
         * @param canonicalName the field canonical name; must not be empty
         * @param mixedCaseName the field mixed case name; must not be empty
         * @param getterName the field getter method name; must not be empty
         * @param setterName the field setter method name; may be {@code null}
         */
        public FieldNameDetails(String canonicalName,
                                String mixedCaseName,
                                String getterName,
                                String setterName) {
            Validate.notEmpty(canonicalName);
            Validate.notEmpty(mixedCaseName);
            Validate.notEmpty(getterName);
            this.canonicalName = canonicalName;
            this.mixedCaseName = mixedCaseName;
            this.getterName = getterName;
            this.setterName = setterName;
        }

        /**
         * Get the field canonical name.
         *
         * @return field canonical name
         */
        public String getCanonicalName() {
            return canonicalName;
        }

        /**
         * Get the field name as defined in the bindings.
         *
         * @return the field mixed case name as defined in the bindings
         */
        public String getMixedCaseName() {
            return mixedCaseName;
        }

        /**
         * Get the binding getter method name for the field.
         *
         * @return getter name
         */
        public String getGetterName() {
            return getterName;
        }

        /**
         * Get the binding setter method name for the field.
         *
         * @return setter name; {@code null}, if the field has no setter
         */
        public String getSetterName() {
            return setterName;
        }
    }

    /**
     * Info about field in this struct type.
     */
    private static class FieldInfo {
        private FieldNameDetails nameDetails;
        private Method getterMethod;
        private Method setterMethod;

        FieldInfo(FieldNameDetails nameDetails, Method getterMethod, Method setterMethod) {
            this.nameDetails = nameDetails;
            this.getterMethod = getterMethod;
            this.setterMethod = setterMethod;
        }

        FieldNameDetails getNameDetails() {
            return nameDetails;
        }

        Method getGetterMethod() {
            return getterMethod;
        }

        Method getSetterMethod() {
            return setterMethod;
        }
    }

    /**
     * Constructor overload for compatibility with bindings generated with
     * older version of vAPI generator.
     */
    // TODO: the new ctor was introduced for the version which follows 2.11.0
    //       this old one (and all the code in runtime and tests) can be removed
    //       in say 2 versioned vapi-core releases (assuming SDKs - NSX, VMC,
    //       vSphere) are all past using the older versions of generator (<= 2.11)
    public StructType(String name,
                      Map<String, Type> fields,
                      Class<?> bindingClass,
                      List<ConstraintValidator> validators,
                      boolean isModel,
                      List<String> modelKeyFields,
                      Map<String, FieldNameDetails> fieldNameDetails) {
        this(name, fields, bindingClass, validators, isModel,
             modelKeyFields, fieldNameDetails, null, null);
    }

    /**
     * Constructor.
     *
     * @param name name of the structure
     * @param fields types of the structure fields
     * @param bindingClass Java language binding of the structure type
     * @param validators validators for constraints defined for in this
     *        {@code StructType}; may be {@code null}
     * @param isModel true iff the structure is marked as model
     * @param modelKeyFields list of the model primary key fields
     * @param fieldNameDetails map of field name details, the key is the field
     *        canonical name, the value is instance of the {@link FieldNameDetails}
     * @param discriminator name of a field which serves as discriminator; may
     *        be {@code null} when no discriminator is present; if not {@code null}
     *        field with such name must be present in {@code fields} map;
     *
     * @param discriminatedBy value identifying this structure when used in a
     *        discriminator field; may be {@code null} if no such value is present;
     *        if not {@code null}, the {@code discriminator} must not be {@code null}
     *        as well
     * @throws IllegalArgumentException if some pre-condition for this method
     *        (described per above) is violated
     */
    public StructType(String name,
                      Map<String, Type> fields,
                      Class<?> bindingClass,
                      List<ConstraintValidator> validators,
                      boolean isModel,
                      List<String> modelKeyFields,
                      Map<String, FieldNameDetails> fieldNameDetails,
                      String discriminator,
                      String discriminatedBy) {
        Validate.notNull(name);
        Validate.notNull(fields);
        Validate.notNull(bindingClass);
        if (discriminatedBy != null && discriminator == null) {
            throw new IllegalArgumentException("Structure with discriminating " +
                        "value has no discriminator field");
        }
        if (discriminator != null && !fields.containsKey(discriminator)) {
            throw new IllegalArgumentException("No such field for discriminator: " +
                        discriminator);
        }

        this.name = name;
        this.discriminatedBy = discriminatedBy;
        this.fields = Collections.unmodifiableMap(fields);
        this.discriminator = discriminator;
        this.bindingClass = bindingClass;
        this.isModel = isModel;
        this.modelKeyFields = modelKeyFields;
        this.validators =
                (validators == null) ? Collections.<ConstraintValidator>emptyList()
                                     : validators;
        this.fieldInfos =
                (fieldNameDetails == null) ? Collections.<String, FieldInfo>emptyMap()
                                           : buildFieldInfoMap(fieldNameDetails);
        if (!this.fieldInfos.isEmpty()) {
            Validate.isTrue(this.fields.keySet()
                    .equals(this.fieldInfos.keySet()));
        }

        fieldWithSetterNames = getFieldsWithSetter(fieldInfos.values());
    }

    private Iterable<String> getFieldsWithSetter(Collection<FieldInfo> collection) {
        for (FieldInfo fieldInfo : collection) {
            if (fieldInfo.getNameDetails().getSetterName() == null) {
                return new Iterable<String>() {
                    @Override
                    public Iterator<String> iterator() {
                        return new SetterAvailableIterator();
                    }
                };
            }
        }
        return this.fieldInfos.keySet();
    }

    private Map<String, FieldInfo> buildFieldInfoMap(Map<String,
                                                     FieldNameDetails> fieldNameDetails) {
        Method[] methods = bindingClass.getMethods();
        Map<String, FieldInfo> result = new HashMap<>();
        for (Map.Entry<String, FieldNameDetails> entry : fieldNameDetails.entrySet()) {
            Method getter = findMethodForName(entry.getValue().getGetterName(), methods);
            logWarnIfMethodNotFound(getter, entry.getValue().getGetterName());

            Method setter = null;
            String setterName = entry.getValue().getSetterName();
            if (setterName != null) {
               setter = findMethodForName(setterName, methods);
               logWarnIfMethodNotFound(setter, setterName);
            }

            result.put(entry.getKey(), new FieldInfo(entry.getValue(), getter, setter));
        }
        return result;
    }

    private void logWarnIfMethodNotFound(Method method, String expectedMethodName) {
        if (method == null) {
            logger.warn("Could not reflectively find getter/setter method {} in class {}",
                        expectedMethodName,
                        bindingClass.getCanonicalName());
        }
    }

    private Method findMethodForName(String methodName, Method[] methods) {
        for (int i = 0; i < methods.length; i++) {
            if (methodName.equals(methods[i].getName())) {
                return methods[i];
            }
        }
        return null;
    }

    /**
     * Returns the name of the structure.
     *
     * @return name of the structure
     */
    public String getName() {
        return name;
    }

    /**
     * Returns the name of the field in this structure which serves as
     * type discriminator.
     *
     * @return the name of the field, or {@code null} if such is not present
     */
    public String getDiscriminator() {
        return this.discriminator;
    }

    /**
     * Returns the value which identifies this structure when used in a
     * discriminator field.
     *
     * @see #getDiscriminator()
     * @return the value identifying this structure, or {@code null} if such is
     *         not available
     */
    public String getDiscriminatedBy() {
        return discriminatedBy;
    }

    /**
     * Returns the names of the structure fields in no particular order.
     *
     * @return names of structure fields
     */
    public Set<String> getFieldNames() {
        return fields.keySet();
    }

    /**
     * Returns the names of the structure fields, which do have setters, in no
     * particular order.
     *
     * @return names of structure fields
     */
    public Iterable<String> getFieldWithSetterNames() {
        return fieldWithSetterNames;
    }

    /**
     * Returns the type of the specified structure field.
     *
     * @param field canonical name of a structure field; must not be null
     * @return type of the structure field
     */
    public Type getField(String field) {
        Validate.notNull(field);
        return fields.get(field);
    }

    /**
     * Returns the type of the specified structure field by its mixed case
     * (java) name.
     *
     * @param javaName mixed case name of a structure field; must not be {@code null}
     * @return type of the structure field or null if there is no field with
     *         the specified name
     */
    public Type getFieldByJavaName(String javaName) {
        Validate.notNull(javaName);
        for (FieldInfo fieldInfo : fieldInfos.values()) {
            if (fieldInfo.getNameDetails().getMixedCaseName().equals(javaName)) {
                return getField(fieldInfo.getNameDetails().getCanonicalName());
            }
        }
        return null;
    }

    /**
     * Get array specifying model key fields names for the model.
     *
     * @return array of field names or null if the structure is not model
     */
    public List<String> getModelKeyFields() {
        if (modelKeyFields == null) {
            return null;
        }
        return Collections.unmodifiableList(modelKeyFields);
    }

    /**
     * Get the {@code Method} representing the getter for the specified field in the
     * the binding class for the structure.
     *
     * @param fieldName name of the field; must not be {@code null}
     * @return the getter method
     * @throws ConverterException if no method is available for the specified field name
     */
    public Method getGetterMethodForField(String fieldName) {
        Validate.notNull(fieldName);
        FieldInfo fieldInfo = getFieldInfo(fieldName);
        if (fieldInfo.getGetterMethod() == null) {
            throw new ConverterException("vapi.bindings.structbinding.struct.missing.getter",
                                         fieldInfo.getNameDetails().getGetterName(),
                                         bindingClass.getClass().getCanonicalName());
        }
        return fieldInfo.getGetterMethod();
    }

    /**
     * Get the {@code Method} representing the setter for the specified field in the
     * the binding class for the structure.
     *
     * @param fieldName name of the field; must not be {@code null}
     * @return the setter method
     * @throws ConverterException if no method is available for the specified field name
     */
    public Method getSetterMethodForField(String fieldName) {
        Validate.notNull(fieldName);
        FieldInfo fieldInfo = getFieldInfo(fieldName);
        if (fieldInfo.getSetterMethod() == null) {
            throw new ConverterException("vapi.bindings.structbinding.struct.missing.setter",
                                         fieldInfo.getNameDetails().getSetterName(),
                                         bindingClass.getCanonicalName());
        }

        return fieldInfo.getSetterMethod();
    }

    FieldInfo getFieldInfo(String fieldName) {
        FieldInfo fieldInfo = fieldInfos.get(fieldName);
        if (fieldInfo == null) {
            throw new ConverterException("vapi.bindings.structbinding.struct.missing.field.details",
                                         fieldName,
                                         bindingClass.getCanonicalName());
        }
        return fieldInfo;
    }

    /**
     * Enforces all static (defined in the IDL) or dynamic constrains defined
     * over this structure. This method will not recursively validate the
     * structures inside the structValue.
     *
     * @param structValue struct value to validate; cannot be null
     * @throws ConstraintValidationException if the constraint is not satisfied
     */
    public void validate(StructValue structValue) {
        Validate.notNull(structValue);
        // TODO the union validator would be called twice for a method call -
        // once by the TypeConverter and second time by the stub/skeleton.
        for (ConstraintValidator validator : validators) {
            validator.validate(structValue);
        }
    }

    /**
     * Enforces all static (defined in the IDL) or dynamic constrains defined
     * over this structure.
     *
     * @param structValue struct value to validate; must not be <code>null</code>
     * @param methodId the method invocation in which context the dataValue was
     *                 passed (as a parameter). must not be <code>null</code>
     * @throws ConstraintValidationException if the constraint is not satisfied
     */
    // TODO: deprecate this method in favor of the other one w/out methodId
    public void validate(StructValue structValue, MethodIdentifier methodId) {
        Validate.notNull(methodId);
        validate(structValue);
    }

    /**
     * Returns a class which represents the Java language binding of this
     * structure.
     *
     * @return language binding class
     */
    public Class<?> getBindingClass() {
        return bindingClass;
    }

    /**
     * Check if the structure is marked as model.
     *
     * @return if the structure is marked as model
     */
    public boolean isModel() {
        return isModel;
    }

    @Override
    public void accept(TypeVisitor visitor) {
        visitor.visit(this);
    }

    /**
     * Get field name details by the field canonical name.
     *
     * @param canonicalName the field canonical name; must not be {@code null}.
     * @return the field name details or {@code null} if no details are available
     */
    public FieldNameDetails getFieldNameDetails(String canonicalName) {
        FieldInfo fieldInfo = fieldInfos.get(canonicalName);
        if (fieldInfo == null) {
            return null;
        }
        return fieldInfo.getNameDetails();
    }

    private class SetterAvailableIterator implements Iterator<String> {
        private Entry<String, FieldInfo> next;
        private final Iterator<Entry<String, FieldInfo>> iterator;

        SetterAvailableIterator() {
            iterator = fieldInfos.entrySet().iterator();
            findNext();
        }

        @Override
        public boolean hasNext() {
            return next != null;
        }

        @Override
        public String next() {
            if (next == null) {
                throw new NoSuchElementException();
            }
            String result = next.getKey();
            findNext();
            return result;
        }

        private void findNext() {
            while (iterator.hasNext()) {
                next = iterator.next();
                if (next.getValue().getNameDetails().getSetterName() != null) {
                    return;
                }
            }
            next = null;
        }

        @Override
        public void remove() {
            throw new UnsupportedOperationException("remove");
        }
    }
}