/* **********************************************************
 * Copyright (c) 2015-2017, 2019 VMware, Inc.  All rights reserved. -- VMware Confidential
 * **********************************************************/

package com.vmware.vapi.client;

import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.Set;

import com.vmware.vapi.bindings.StubConfiguration;
import com.vmware.vapi.internal.client.ApiClientImpl;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.HttpConfiguration;

/**
 * Configuration for {@link ApiClientImpl}s.
 *
 * <p>Use {@link Builder} or the static factory methods to create instances of
 * this class.
 */
public final class Configuration {
    private Map<String, Object> configProps;

    public static final String HTTP_CONFIG_CFG = "http.client.config";

    public static final String STUB_CONFIG_CFG = "vapi.stub.config";

    /**
     * Constructor.
     *
     * @param props configuration properties to be registered with this instance
     */
    protected Configuration(Map<String, Object> props) {
        Validate.notNull(props);
        configProps = new HashMap<>(props);
    }

    /**
     * Returns the names of all configuration properties registered with this instance.
     *
     * @return set of all registered configuration property names
     */
    public Set<String> properties() {
        return Collections.unmodifiableSet(configProps.keySet());
    }

    /**
     * Get the value of given configuration property.
     *
     * <p>The type of the value must be specified, and the returned reference is of that type.
     *
     * @param propName name of the property to fetch; must not be {@code null}
     * @param propClass expected type of the property value; must not be {@code null}
     * @return value of the property as strongly typed reference or {@code null} if no
     *         property is registered for {@code propName}
     * @throws ClassCastException if property is found but its value doesn't match the expected
     *         type specified by {@code propClass}
     */
    public <T> T getProperty(String propName, Class<T> propClass) {
        Validate.notNull(propName);
        Validate.notNull(propClass);

        Object value = configProps.get(propName);
        if (value == null) {
            return null;
        }

        if (!propClass.isAssignableFrom(value.getClass())) {
            throw new ClassCastException("Unexpected class for property: " + propName);
        }

        return propClass.cast(value);
    }

    protected Map<String, Object> getAllProperties() {
        return Collections.unmodifiableMap(configProps);
    }

    /**
     * Creates a new empty configuration.
     *
     * @return configuration which has no settings
     */
    public static Configuration newEmptyConfig() {
        return new Builder().build();
    }

    /**
     * Creates a new configuration instance which holds HTTP settings. This
     * includes TCP timeouts and SSL config.
     *
     * @param httpConfig HTTP settings to use
     * @return configuration with the specified HTTP settings
     */
    public static Configuration newHttpConfig(HttpConfiguration httpConfig) {
        return new Builder().register(HTTP_CONFIG_CFG, httpConfig).build();
    }

    /**
     * Creates a new configuration instance which holds stub configuration.
     *
     * @param stubConfig stub settings to use
     * @return configuration with the specified stub settings
     */
    public static Configuration newStubConfig(StubConfiguration stubConfig) {
        return new Builder().register(STUB_CONFIG_CFG, stubConfig).build();
    }

    /**
     * Builder for {@link Configuration} instances.
     *
     * <p>This class also provides static factory methods for easy creation of
     * commonly used {@link Configuration}s.
     */
    public final static class Builder {
        private Map<String, Object> configProps;

        /**
         * Constructor.
         */
        public Builder() {
            this(new HashMap<String, Object>());
        }

        /**
         * Constructor.
         *
         * <p>Initializes the newly constructed instance with properties of the
         * specified configuration.
         *
         * @param config configuration to provide initial properties for this
         *                   instance; muse not be {@code null}
         */
        public Builder(Configuration config) {
            this(config.getAllProperties());
        }

        /**
         * Constructor.
         *
         * <p>Initializes the newly constructed instance with the specified properties.
         *
         * @param properties initial configuration properties; must not be {@code null}
         */
        public Builder(Map<String, Object> properties) {
            Validate.notNull(properties);
            configProps = new HashMap<>(properties);
        }

        /**
         * Registers a setting for the configuration to be built.
         *
         * @param propName name of the setting
         * @param value value of the setting
         * @return this
         */
        public Builder register(String propName, Object value) {
            Validate.notNull(propName);
            Validate.notNull(value);
            this.configProps.put(propName, value);
            return this;
        }

        public Configuration build() {
            return new Configuration(configProps);
        }
    }
}
