/* **********************************************************
 * Copyright 2012-2013, 2019 VMware, Inc.  All rights reserved.
 *      -- VMware Confidential
 * **********************************************************/

/*
 * SpringConfigurator.java --
 *
 *      Creates vAPI runtime objects from a Spring config/application context.
 */
package com.vmware.vapi.config;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.List;
import java.util.Properties;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.config.PropertyPlaceholderConfigurer;
import org.springframework.context.support.AbstractRefreshableConfigApplicationContext;
import org.springframework.context.support.ClassPathXmlApplicationContext;

import com.vmware.vapi.MessageFactory;
import com.vmware.vapi.core.ApiProvider;
import com.vmware.vapi.internal.util.Validate;
import com.vmware.vapi.protocol.ProtocolHandler;
import com.vmware.vapi.provider.aggregator.ApiAggregator;
import com.vmware.vapi.provider.local.LocalProvider;

/**
 * Creates vAPI runtime objects from a Spring config/application context.
 */
public class SpringConfigurator implements Configurator {
    private static final String DEFAULT_TC_SERVER_CONFIG = "vapi-tcserver-spring.xml";
    private static final String DEFAULT_HTTP_CONFIG = "vapi-http-spring.xml";
    private static final String DEFAULT_HTTPS_CONFIG = "vapi-https-spring.xml";

    private static final String PROVIDER_BEAN_ID = "apiProvider";
    private static final String HTTP_SERVER_BEAN_ID = "httpOnlyServer";
    private static final String HTTPS_SERVER_BEAN_ID = "httpsOnlyServer";
    private static final String HTTP_HTTPS_SERVER_BEAN_ID = "httpAndHttpsServer";

    private final static Logger logger =
            LoggerFactory.getLogger(SpringConfigurator.class);

    private AbstractRefreshableConfigApplicationContext ctx;
    private URL httpUrl;
    private URL httpsUrl;

    /**
     * Constructor.
     *
     * <p>Creates an instance that load <tt>Spring</tt> application context
     * and configures the infrastructure based on the specified configuration
     * files.
     *
     * <p>It is expected that in the provided configuration
     * <ul><li>exactly 1 bean exists with <code>id="apiProvider"</code> and
     *         that bean is of type {@link LocalProvider} or
     *         {@link ApiAggregator}</li>
     *     <li>1 or more beans of types descendant of {@link ProtocolHandler}
     *         exists</li>
     * </ul>
     * The above beans are the result from <code>Configurator</code> interface
     * methods implemented by this class.
     *
     * @param configFiles  <tt>Spring</tt> XML configuration files, must be
     *                     available on the classpath
     * @throws IllegalArgumentException  if <code>configFiles</code> is
     *         <code>null</code> or empty
     */
    public SpringConfigurator(List<String> configFiles) {
        Validate.notEmpty(configFiles);

        loadSpringContext(configFiles,
                          null);   // no Properties to be used in XML config
    }

    /**
     * Constructor.
     *
     * <p>Creates an instance that uses the specified <code>uri</code> and
     * optionally <code>providerSpringConfig</code> to configure the
     * infrastructure.
     *
     * <p>Instances created with this constructor uses <b>default</b>
     * configuration for JSON/HTTP <code>ProtocolHandler<code>. HTTP server
     * uses HTTP listener only (no HTTPS).
     * The configuration is available from:
     * <ul>
     *    <li>vapi-tcserver-spring.xml</li>
     *    <li>vapi-http-spring.xml</li>
     * </ul>
     *
     * <p>Custom <tt>Spring</tt> XML configuration may be provided with the
     * <code>providerSpringConfig</code>. If provided it will be loaded and a
     * bean with id <code>id="apiProvider"</code> will be used by this
     * configurator.
     *
     * @param uri  HTTP URL for server to listen on
     * @param providerSpringConfig  <tt>Spring</tt> XML configuration files
     *        supposed to contain definition for <code>ApiProvider</code> to
     *        be used' must be available on the classpath; must not be
     *        <code>null</code>
     *
     * @see SpringConfigurator#SpringConfigurator(List)
     */
    public SpringConfigurator(String uri,
                              String providerSpringConfig) {
        Validate.notNull(providerSpringConfig);
        httpUrl = parseUrl(uri, "http");

        List<String> configs = new ArrayList<String>();
        configs.add(DEFAULT_TC_SERVER_CONFIG);
        configs.add(DEFAULT_HTTP_CONFIG);
        configs.add(providerSpringConfig);

        Properties configProps = new Properties();
        prepareHttpConfigProperties(httpUrl, configProps);
        configProps.put("httpServer", HTTP_SERVER_BEAN_ID);

        loadSpringContext(configs, configProps);
    }

    /**
     * Constructor.
     *
     * <p>Creates an instance that uses the specified <code>uri</code> and
     * optionally <code>providerSpringConfig</code> to configure the
     * infrastructure.
     *
     * <p>Instances created with this constructor uses <b>default</b>
     * configuration for JSON/HTTP <code>ProtocolHandler<code>. HTTP server
     * uses HTTPS listener optionally HTTP one.
     * The configuration is available from:
     * <ul>
     *    <li>vapi-tcserver-spring.xml</li>
     *    <li>vapi-http-spring.xml</li>
     *    <li>vapi-https-spring.xml</li>
     * </ul>
     *
     * @param uri          HTTP URL for server to listen on
     * @param secureUri    HTTPS URL for server to listen on
     * @param keystore     SSL keystore
     * @param password     password for the SSL keystore
     * @param keypassword  password for the SSL key
     * @param providerSpringConfig  <tt>Spring</tt> XML configuration files
     *        supposed to contain definition for <code>ApiProvider</code> to
     *        be used' must be available on the classpath; must not be
     *        <code>null</code>
     *
     * @see SpringConfigurator#SpringConfigurator(String, String)
     * @see SpringConfigurator#SpringConfigurator(List)
     */
    public SpringConfigurator(String uri,
                              String secureUri,
                              String keystore,
                              String password,
                              String keypassword,
                              String providerSpringConfig) {
        Validate.notNull(providerSpringConfig);
        httpUrl = parseUrl(uri, "http");
        httpsUrl = parseUrl(secureUri, "https");

        Properties configProps = prepareConfigProperties(httpUrl, httpsUrl,
                keystore, password, keypassword);

        // Load HTTP/HTTPS configuration from spring xml config file.
        List<String> configs = new ArrayList<String>();
        configs.add(DEFAULT_TC_SERVER_CONFIG);
        configs.add(DEFAULT_HTTPS_CONFIG);
        if (httpUrl != null) {
            configs.add(DEFAULT_HTTP_CONFIG);
        }
        configs.add(providerSpringConfig);

        String serverBeanId = httpUrl == null ? HTTPS_SERVER_BEAN_ID
                                              : HTTP_HTTPS_SERVER_BEAN_ID;
        configProps.put("httpServer", serverBeanId);

        loadSpringContext(configs, configProps);
    }

    private void loadSpringContext(List<String> configs,
                                   Properties configProps) {

        ctx = new ClassPathXmlApplicationContext();
        if (configProps != null) {
            PropertyPlaceholderConfigurer propsPostProcessor =
                    new PropertyPlaceholderConfigurer();
            propsPostProcessor.setProperties(configProps);

            if (logger.isDebugEnabled()) {
                logger.debug("Loading Spring configs files: " + configs);
            }
            ctx.addBeanFactoryPostProcessor(propsPostProcessor);
        }
        ctx.setConfigLocations(configs.toArray(new String[configs.size()]));
        ctx.refresh();
    }

    private URL parseUrl(String url, String urlType) {
        if (url == null) {
            return null;
        }

        try {
            return new URL(url);
        } catch (MalformedURLException ex) {
            String msg =
                MessageFactory.getMessage("vapi.http.invalidarguments",
                                          urlType + " url").toString();
            throw new RuntimeException(msg);
        }
    }

    /**
     * Prepares <code>Properties</code> instance with configuration properties
     * to be used for post-processing of bean definitions in Spring
     * configuration.
     */
    private Properties prepareConfigProperties(URL httpUrl,
                                               URL httpsUrl,
                                               String keystore,
                                               String password,
                                               String keyPassword) {
        Properties result = new Properties();
        if (httpUrl != null) {
            prepareHttpConfigProperties(httpUrl, result);
        }
        if (httpsUrl != null) {
            prepareSecureHttpConfigProperties(httpUrl, result);
        }

        String invalidArgs = null;
        if (keystore == null) {
            invalidArgs = "keystore path";
        } else if (password == null) {
            invalidArgs = "keystore password";
        }

        if (invalidArgs != null) {
            // TODO: this is not the right way to use Message: toString() it
            String msg = MessageFactory.getMessage(
                    "vapi.http.invalidarguments", invalidArgs).toString();
            throw new RuntimeException(msg);
        }
        result.setProperty("keystore.path", keystore);
        result.setProperty("keystore.password", password);
        if (keyPassword != null) {
            result.setProperty("key.password", keyPassword);
        }

        return result;
    }

    private void prepareHttpConfigProperties(URL httpUrl, Properties props) {
        props.setProperty("http.port", String.valueOf(httpUrl.getPort()));
        props.setProperty("http.path", httpUrl.getPath());
    }

    private void prepareSecureHttpConfigProperties(URL httpUrl, Properties props) {
        props.setProperty("https.port", String.valueOf(httpsUrl.getPort()));
        props.setProperty("https.path", httpsUrl.getPath());
    }

    @Override
    public ApiProvider getApiProvider() {
        return ctx.getBean(PROVIDER_BEAN_ID, ApiProvider.class);
    }

    @Override
    public List<ProtocolHandler> getProtocolHandlers() {
        return new ArrayList<ProtocolHandler>(
                ctx.getBeansOfType(ProtocolHandler.class).values());
    }

}
